package ru.yandex.solomon.alert.cluster.server.grpc;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import io.grpc.Status;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertInterner;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.protobuf.EvaluationAssignmentKey;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage.AssignEvaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage.UnassignEvaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.Evaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.EvaluationError;
import ru.yandex.solomon.alert.protobuf.TAssignmentSeqNo;
import ru.yandex.solomon.alert.rule.EvaluationState;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcEvaluationStreamConverter {
    public static Alert toAlert(AssignEvaluation assign) {
        return switch (assign.getTypeCase()) {
            case ALERT -> AlertConverter.protoToAlert(assign.getAlert(), AlertInterner.I);
            case SUB_ALERT -> AlertConverter.protoToSubAlert(assign.getSubAlert(), AlertInterner.I);
            default -> throw new StatusRuntimeExceptionNoStackTrace(Status.UNIMPLEMENTED.withDescription("Unsupported assign type: " + assign.getTypeCase()));
        };
    }

    @Nullable
    public static EvaluationState toState(AssignEvaluation assign) {
        return AlertConverter.protoToState(assign.getLatestState());
    }

    @Nullable
    public static EvaluationState toState(Evaluation evaluation) {
        return AlertConverter.protoToState(evaluation.getState());
    }

    public static Evaluation toEvaluation(EvaluationAssignmentKey key, EvaluationState state) {
        return Evaluation.newBuilder()
                .setAssignmentKey(key)
                .setState(AlertConverter.stateToProto(state))
                .build();
    }

    public static AssignEvaluation toAssignEvaluation(EvaluationAssignmentKey key, Alert alert, @Nullable EvaluationState state) {
        var result = AssignEvaluation.newBuilder()
                .setAssignmentKey(key)
                .setLatestState(AlertConverter.stateToProto(state));

        if (alert instanceof SubAlert) {
            result.setSubAlert(AlertConverter.subAlertToProto((SubAlert) alert));
        } else {
            result.setAlert(AlertConverter.alertToProto(alert));
        }

        return result.build();
    }

    public static UnassignEvaluation toUnassignEvaluation(EvaluationAssignmentKey key) {
        return UnassignEvaluation.newBuilder()
                .setAssignKey(key)
                .build();
    }


    public static UnassignEvaluation toUnassignEvaluation(TAssignmentSeqNo key) {
        return UnassignEvaluation.newBuilder()
                .setAssignKey(EvaluationAssignmentKey.newBuilder()
                        .setAssignGroupId(key)
                        .build())
                .build();
    }

    public static EvaluationAssignmentKey toKey(TAssignmentSeqNo groupId, long assignId) {
        return EvaluationAssignmentKey.newBuilder()
                .setAssignGroupId(groupId)
                .setAssignId(assignId)
                .build();
    }

    public static EvaluationError toError(EvaluationAssignmentKey key, Status status) {
        return EvaluationError.newBuilder()
                .setAssignmentKey(key)
                .setStatus(toStatus(status))
                .build();
    }

    public static EvaluationError toError(TAssignmentSeqNo key, Status status) {
        return toError(EvaluationAssignmentKey.newBuilder()
                .setAssignGroupId(key)
                .build(), status);
    }

    public static com.google.rpc.Status toStatus(Status status) {
        var result = com.google.rpc.Status.newBuilder();
        result.setCode(status.getCode().value());
        if (!Strings.isNullOrEmpty(status.getDescription())) {
            result.setMessage(status.getDescription());
        }

        return result.build();
    }

    public static Status toStatus(com.google.rpc.Status status) {
        var result = Status.fromCodeValue(status.getCode());
        if (Strings.isNullOrEmpty(status.getMessage())) {
            return result;
        }

        return result.withDescription(status.getMessage());
    }
}
