package ru.yandex.solomon.alert.cluster.server.grpc.evaluation;

import javax.annotation.concurrent.NotThreadSafe;

import io.grpc.stub.StreamObserver;

import static com.google.common.base.Preconditions.checkState;

/**
 * @author Vladimir Gordiychuk
 */
@NotThreadSafe
public abstract class BatchWriter<T> {
    static final int MAX_BATCH_SIZE_IN_BYTES = 50 << 20; // 50 MiB

    private final StreamObserver<T> output;
    private final int maxBatchSize;
    private int size;
    private boolean closed;

    public BatchWriter(StreamObserver<T> output, int maxBatchSize) {
        this.output = output;
        this.maxBatchSize = maxBatchSize;
    }

    public void onError(Throwable t) {
        checkState(!closed, "stream already closed");
        closed = true;
        output.onError(t);
    }

    public void onCompleted() {
        checkState(!closed, "stream already closed");
        closed = true;
        output.onCompleted();
    }

    public void flush() {
        if (size == 0) {
            return;
        }

        try {
            checkState(!closed, "stream already closed");
            output.onNext(build());
        } catch (Throwable e) {
            try {
                onError(e);
            } catch (Throwable e2) {
                e.addSuppressed(e2);
            }
            throw e;
        } finally {
            size = 0;
        }
    }

    public boolean isClosed() {
        return closed;
    }

    protected abstract T build();

    protected void add(int size) {
        this.size += size;
        flushIfNeeded();
    }

    private void flushIfNeeded() {
        if (size >= maxBatchSize) {
            flush();
        }
    }
}
