package ru.yandex.solomon.alert.cluster.server.grpc.evaluation;

import java.time.Duration;
import java.util.ArrayList;
import java.util.IntSummaryStatistics;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Flow.Subscriber;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

/**
 * @author Vladimir Gordiychuk
 */
public class ClientEvaluationCluster implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ClientEvaluationCluster.class);

    private final ClusterDiscovery<GrpcTransport> discovery;
    private final int maxStream;
    private final ScheduledExecutorService timer;
    private final Executor executor;
    private final Metrics metrics;
    private final ActorRunner actor;
    private final RebalanceProcess rebalanceProcess;

    // input
    private final ArrayListLockQueue<ClientAssignReq> outboundAssign = new ArrayListLockQueue<>();

    // state
    private final AtomicReference<CompletableFuture<Void>> actFuture = new AtomicReference<>();
    private final ConcurrentMap<String, ClientEvaluationNode> nodeByFqdn = new ConcurrentHashMap<>();
    private final List<ClientEvaluationNode> readyNodes = new ArrayList<>();
    private final List<NodeWeight> assignCandidates = new ArrayList<>();
    private int assignCandidateSize = 0;
    private volatile boolean closed;

    public ClientEvaluationCluster(
            ClusterDiscovery<GrpcTransport> discovery,
            int maxStream,
            ScheduledExecutorService timer,
            Executor executor,
            MetricRegistry registry)
    {
        this.discovery = discovery;
        this.maxStream = maxStream;
        this.timer = timer;
        this.executor = executor;
        this.metrics = new Metrics(registry);
        this.actor = new ActorRunner(this::act, executor);
        this.rebalanceProcess = new RebalanceProcess();
        this.onDiscoveryChanged();
    }

    private void onDiscoveryChanged() {
        discovery.callbackOnChange(this::onDiscoveryChanged);
        actor.schedule();
    }

    public void assign(ClientAssignReq assignReq) {
        outboundAssign.enqueue(assignReq);
        actor.schedule();
    }

    public void setActive(String fqdn, boolean flag) {
        var node = nodeByFqdn.get(fqdn);
        if (node != null) {
            node.setActive(flag);
            return;
        }

        if (discovery.hasNode(fqdn)) {
            actor.schedule();
        }
    }

    public CompletableFuture<Void> awaitAct() {
        return actFuture.updateAndGet(prev -> {
            if (prev != null) {
                return prev;
            }
            return new CompletableFuture<>();
        });
    }

    public void scheduleAct() {
        actor.schedule();
    }

    private void act() {
        var doneFuture = actFuture.getAndSet(null);
        try {
            if (closed) {
                processClose();
                return;
            }

            actualizeNodes();
            actualizeReadyNodes();
            actualizeNodesToAssign();
            processAssignments();
        } finally {
            if (doneFuture != null) {
                doneFuture.complete(null);
            }
        }
    }

    private void actualizeNodes() {
        var actual = discovery.getNodes();
        addNewNodes(actual);
        removeOldNodes(actual);
    }

    private void addNewNodes(Set<String> actual) {
        for (var fqdn : actual) {
            var node = nodeByFqdn.get(fqdn);
            if (node != null) {
                continue;
            }

            var transport = transportOrNull(fqdn);
            if (transport == null) {
                continue;
            }

            node = new ClientEvaluationNode(transport, maxStream, timer, executor);
            logger.info("Node {} included into evaluation cluster", node);
            nodeByFqdn.put(fqdn, node);
        }
    }

    private void removeOldNodes(Set<String> fresh) {
        var it = nodeByFqdn.entrySet().iterator();
        while (it.hasNext()) {
            var entry = it.next();
            if (!fresh.contains(entry.getKey())) {
                logger.info("Node {} excluded from evaluation cluster", entry.getKey());
                quiteClose(entry.getValue());
                it.remove();
            }
        }
    }

    private void processAssignments() {
        if (outboundAssign.size() == 0) {
            return;
        }

        if (assignCandidateSize == 0) {
            // await next update
            return;
        }

        var requests = outboundAssign.dequeueAll();
        var random = ThreadLocalRandom.current();
        for (var req : requests) {
            try {
                var candidate = assignCandidates.get(random.nextInt(0, assignCandidateSize));
                var node = candidate.node;
                logger.info("Assign {} to {}", req.alert().getKey(), node);
                node.assign(req);
            } catch (Throwable e) {
                logger.warn("failed assign {}", req, e);
                quiteClose(req.subscriber());
            }
        }
        logger.info("Assigned {} to evaluate", requests.size());
    }

    private void actualizeReadyNodes() {
        readyNodes.clear();
        for (var node : nodeByFqdn.values()) {
            if (node.isConnected() && node.isActive()) {
                readyNodes.add(node);
            }
        }
    }

    private void actualizeNodesToAssign() {
        try {
            assignCandidates.clear();
            for (var node : readyNodes) {
                assignCandidates.add(new NodeWeight(node, node.getEvaluationStatus()));
            }

            assignCandidates.sort(NodeWeight::compareTo);
            int from = 0;
            while (from < assignCandidates.size()) {
                if (assignCandidates.get(from).status.assignments() != 0) {
                    break;
                }

                from++;
            }

            int withUsage = assignCandidates.size() - from;
            if (withUsage <= 2) {
                assignCandidateSize = assignCandidates.size();
                return;
            }

            int half = withUsage / 2;
            assignCandidateSize = from + half;
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

    private void processClose() {
        // close pending assign
        var requests = outboundAssign.dequeueAll();
        for (var req : requests) {
            quiteClose(req.subscriber());
        }

        // close node
        for (var node : nodeByFqdn.values()) {
            quiteClose(node);
        }
        nodeByFqdn.clear();
    }

    private void quiteClose(AutoCloseable node) {
        try {
            node.close();
        } catch (Exception e) {
            logger.warn("Unable to close {}", node);
        }
    }

    private GrpcTransport transportOrNull(String fqdn) {
        try {
            return discovery.getTransportByNode(fqdn);
        } catch (Throwable e) {
            return null;
        }
    }

    private void quiteClose(Subscriber<?> subscriber) {
        try {
            subscriber.onSubscribe(NoopSubscription.INSTANCE);
        } catch (Throwable e) {
            logger.warn("subscriber={} failed on subscribe", subscriber, e);
        }

        try {
            subscriber.onComplete();
        } catch (Throwable e) {
            logger.warn("subscriber={}, failed on complete", subscriber, e);
        }
    }

    @Override
    public void close() {
        closed = true;
        rebalanceProcess.close();
        actor.schedule();
    }

    record NodeWeight(ClientEvaluationNode node, NodeEvaluationStatus status) implements Comparable<NodeWeight> {
        @Override
        public int compareTo(NodeWeight that) {
            var compare = Integer.compare(status.assignments(), that.status.assignments());
            if (compare != 0) {
                return compare;
            }

            return Double.compare(status.cpuNanos(), that.status.cpuNanos());
        }
    }

    private final class RebalanceProcess implements AutoCloseable {
        private double dispersionThreshold = 0.05;
        private double unassignPercent = 0.01;
        @InstantMillis
        private long rebalancedAt;

        private final PingActorRunner actor;

        public RebalanceProcess() {
            this.actor = PingActorRunner.newBuilder()
                    .executor(executor)
                    .timer(timer)
                    .pingInterval(Duration.ofSeconds(15))
                    .backoffDelay(Duration.ofSeconds(5))
                    .backoffMaxDelay(Duration.ofMinutes(5))
                    .operation("evaluation_rebalance")
                    .onPing(ignore -> {
                        act();
                        return CompletableFuture.completedFuture(null);
                    })
                    .build();
            actor.schedule();
        }

        public void act() {
            var assignmentsStats = assignmentStatistics();
            metrics.nodeCount.set(assignmentsStats.getCount());
            if (assignmentsStats.getCount() <= 1) {
                return;
            }

            double dispersion = (double) (assignmentsStats.getMax() - assignmentsStats.getMin()) / (double) assignmentsStats.getMax();
            metrics.dispersion.set(dispersion);
            if (Double.isNaN(dispersion) || dispersion <= dispersionThreshold) {
                return;
            }

            double average = assignmentsStats.getAverage();
            metrics.expectedCount.set(average);
            logger.info("Evaluation dispersion {} expected evaluation per node {}, node count {}",
                    dispersion, average, assignmentsStats.getCount());
            for (var node : nodeByFqdn.values()) {
                int assignmentsOnNode = node.getEvaluationStatus().assignments();
                if (assignmentsOnNode < average) {
                    node.cancelUnassign();
                    continue;
                }

                int unassignCount = (int) Math.round((assignmentsOnNode - average) * unassignPercent);
                metrics.unassigned.add(unassignCount);
                node.unassign(unassignCount);
                logger.info("Unassign {} evaluation's from node {} as rebalance process", unassignCount, node);
            }

            rebalancedAt = System.currentTimeMillis();
        }

        @ManagerMethod
        public void setDispersionThreshold(double dispersionThreshold) {
            this.dispersionThreshold = dispersionThreshold;
        }

        @ManagerMethod
        public void setUnassignPercent(double unassignPercent) {
            this.unassignPercent = unassignPercent;
        }

        private IntSummaryStatistics assignmentStatistics() {
            var timeThreshold = System.currentTimeMillis() - TimeUnit.MINUTES.toMillis(1L);
            var summary = new IntSummaryStatistics();
            for (var node : nodeByFqdn.values()) {
                if (!node.isConnected()) {
                    continue;
                }

                if (!node.isActive()) {
                    continue;
                }

                var status = node.getEvaluationStatus();
                if (status.receivedAt() < timeThreshold) {
                    continue;
                }

                summary.accept(status.assignments());
            }

            return summary;
        }

        @Override
        public void close() {
            actor.close();
        }
    }

    private static class Metrics {
        private final GaugeDouble dispersion;
        private final GaugeDouble expectedCount;
        private final GaugeInt64 nodeCount;
        private final Rate unassigned;

        public Metrics(MetricRegistry registry) {
            dispersion = registry.gaugeDouble("client.evaluation.balancer.dispersion");
            expectedCount = registry.gaugeDouble("client.evaluation.balancer.expected_count");
            nodeCount = registry.gaugeInt64("client.evaluation.balancer.nodes_count");
            unassigned = registry.rate("client.evaluation.balancer.unassigned");
        }
    }
}
