package ru.yandex.solomon.alert.cluster.server.grpc.evaluation;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Flow.Subscriber;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.alert.protobuf.EvaluationServerStatusRequest;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage;
import ru.yandex.solomon.alert.protobuf.TAlertingClusterServiceGrpc;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

import static com.google.common.base.Preconditions.checkState;

/**
 * @author Vladimir Gordiychuk
 */
public class ClientEvaluationNode implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ClientEvaluationNode.class);

    private final GrpcTransport transport;
    private final int maxStream;
    private final Executor executor;
    private final ActorRunner actor;
    private final PingActorRunner serverStatusActor;

    private final AtomicReference<CompletableFuture<Void>> actFuture = new AtomicReference<>();
    private final ArrayListLockQueue<ClientAssignReq> outboundAssign = new ArrayListLockQueue<>();
    private final Int2ObjectMap<ClientEvaluationStream> streamById = new Int2ObjectOpenHashMap<>();
    private int nextStreamId;
    private final ClientEvaluationBreakerImpl evaluationBreaker = new ClientEvaluationBreakerImpl();
    private final AtomicBoolean active = new AtomicBoolean(true);
    private volatile NodeEvaluationStatus evaluationStatus = NodeEvaluationStatus.EMPTY;
    private volatile boolean closed;

    public ClientEvaluationNode(GrpcTransport transport, int maxStream, ScheduledExecutorService timer, Executor executor) {
        this.transport = transport;
        this.maxStream = maxStream;
        this.executor = executor;
        this.actor = new ActorRunner(this::act, executor);
        this.serverStatusActor = PingActorRunner.newBuilder()
                .executor(executor)
                .timer(timer)
                .pingInterval(Duration.ofSeconds(15))
                .backoffDelay(Duration.ofSeconds(1))
                .backoffMaxDelay(Duration.ofMinutes(5))
                .operation("evaluation_server_status_on_" + transport.getAddress())
                .onPing(this::actEvaluationServerStatus)
                .build();
        serverStatusActor.forcePing();
    }

    public boolean isConnected() {
        return transport.isConnected();
    }

    public boolean isActive() {
        return active.get();
    }

    public void assign(ClientAssignReq assignReq) {
        outboundAssign.enqueue(assignReq);
        actor.schedule();
    }

    public void setActive(boolean active) {
        evaluationBreaker.setActive(active);
        if (this.active.compareAndSet(!active, active)) {
            actor.schedule();
        }
    }

    public void unassign(int count) {
        evaluationBreaker.unassign(count);
    }

    public void cancelUnassign() {
        evaluationBreaker.cancelUnassign();
    }

    public NodeEvaluationStatus getEvaluationStatus() {
        return evaluationStatus;
    }

    public CompletableFuture<Void> awaitAct() {
        return actFuture.updateAndGet(prev -> {
            if (prev != null) {
                return prev;
            }
            return new CompletableFuture<>();
        });
    }

    private void act() {
        var doneFuture = actFuture.getAndSet(null);
        try {
            if (closed) {
                processClose();
                return;
            }

            closeNotActualStreams();
            processAssignments();
        } finally {
            if (doneFuture != null) {
                doneFuture.complete(null);
            }
        }
    }

    private void processAssignments() {
        if (outboundAssign.size() == 0) {
            return;
        }

        var requests = outboundAssign.dequeueAll();
        if (!active.get()) {
            for (var req : requests) {
                quiteClose(req.subscriber());
            }
            return;
        }

        for (var req : requests) {
            try {
                var stream = findLeastLoadedStream();
                stream.assign(req);
            } catch (Throwable e) {
                logger.warn("node={} failed assign {}", transport, req, e);
                quiteClose(req.subscriber());
            }
        }
    }

    private void processClose() {
        var requests = outboundAssign.dequeueAll();
        for (var req : requests) {
            quiteClose(req.subscriber());
        }

        for (var stream : streamById.values()) {
            stream.close();
        }
        streamById.clear();
    }

    private ClientEvaluationStream findLeastLoadedStream() {
        if (streamById.size() < maxStream) {
            return makeStream();
        }

        int minSize = Integer.MAX_VALUE;
        ClientEvaluationStream result = null;
        var it = streamById.values().iterator();
        while (it.hasNext()) {
            var stream = it.next();
            if (stream.isDone()) {
                it.remove();
                return makeStream();
            }

            int size = stream.size();
            if (size < minSize) {
                minSize = size;
                result = stream;
            }
        }

        if (result == null) {
            return makeStream();
        }

        return result;
    }

    private void quiteClose(Subscriber<?> subscriber) {
        try {
            subscriber.onSubscribe(NoopSubscription.INSTANCE);
        } catch (Throwable e) {
            logger.warn("node={}, subscriber={} failed on subscribe", transport, subscriber, e);
        }

        try {
            subscriber.onComplete();
        } catch (Throwable e) {
            logger.warn("node={}, subscriber={}, failed on complete", transport, subscriber, e);
        }
    }

    private void closeNotActualStreams() {
        var it = streamById.values().iterator();
        while (it.hasNext()) {
            var stream = it.next();
            if (stream.isDone()) {
                it.remove();
            } else if (stream.size() == 0) {
                stream.close();
                it.remove();
            }
        }
    }

    private CompletableFuture<Void> actEvaluationServerStatus(int attempt) {
        if (!isConnected()) {
            return CompletableFuture.completedFuture(null);
        }

        var req = EvaluationServerStatusRequest.getDefaultInstance();
        var deadline = System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(5);
        return transport.unaryCall(TAlertingClusterServiceGrpc.getEvaluationServerStatusMethod(), req, deadline)
                .thenAccept(response -> {
                    var host = transport.getAddress().getHost();
                    if (!host.equals(response.getNode())) {
                        throw new StatusRuntimeExceptionNoStackTrace(Status.FAILED_PRECONDITION
                                .withDescription("expected fqdn " + host + " but was " + response.getNode()));
                    }

                    evaluationStatus = NodeEvaluationStatus.of(host, response);
                });
    }

    private ClientEvaluationStream makeStream() {
        var halfInitOutput = new ClientMessageStreamObserver();
        var stream = new ClientEvaluationStream(halfInitOutput, executor, evaluationBreaker);
        var output = transport.bidiStreamingCall(TAlertingClusterServiceGrpc.getEvaluationStreamMethod(), stream);
        halfInitOutput.init(output);
        streamById.put(nextStreamId++, stream);
        return stream;
    }

    @Override
    public void close() {
        closed = true;
        serverStatusActor.close();
        actor.schedule();
    }

    @Override
    public String toString() {
        return transport.toString();
    }

    private static class ClientMessageStreamObserver implements StreamObserver<EvaluationStreamClientMessage> {
        private volatile StreamObserver<EvaluationStreamClientMessage> delegate;

        private synchronized void init(StreamObserver<EvaluationStreamClientMessage> delegate) {
            checkState(this.delegate == null, "already initialized");
            this.delegate = delegate;
        }

        @Override
        public void onNext(EvaluationStreamClientMessage value) {
            var delegate = this.delegate;
            checkState(delegate != null, "stream not initialized yet");
            delegate.onNext(value);
        }

        @Override
        public void onError(Throwable t) {
            var delegate = this.delegate;
            checkState(delegate != null, "stream not initialized yet");
            delegate.onError(t);
        }

        @Override
        public void onCompleted() {
            var delegate = this.delegate;
            checkState(delegate != null, "stream not initialized yet");
            delegate.onCompleted();
        }
    }

}
