package ru.yandex.solomon.alert.cluster.server.grpc.evaluation;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Flow;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import io.netty.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.alert.protobuf.EvaluationAssignmentKey;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage;
import ru.yandex.solomon.alert.util.Async;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toAssignEvaluation;
import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toUnassignEvaluation;

/**
 * @author Vladimir Gordiychuk
 */
public class ClientEvaluationStream implements StreamObserver<EvaluationStreamServerMessage>, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ClientEvaluationStream.class);

    static final long NO_MESSAGE_TIMEOUT_MS = TimeUnit.MINUTES.toMillis(2);

    private final EvaluationStreamClientMessageWriter output;
    private final ClientEvaluationBreaker evaluationBreaker;
    private final long noMessageTimeoutMs;
    private final ActorRunner actor;

    // Inbound
    private final ArrayListLockQueue<EvaluationStreamServerMessage> inboundMessages = new ArrayListLockQueue<>();
    private final CompletableFuture<Status> inboundDone = new CompletableFuture<>();

    // Outbound
    private final ArrayListLockQueue<ClientAssignReq> outboundAssign = new ArrayListLockQueue<>();
    private final ArrayListLockQueue<EvaluationAssignmentKey> outboundUnassign = new ArrayListLockQueue<>();
    private final CompletableFuture<Status> outboundDone = new CompletableFuture<>();

    // State
    private volatile int size;
    private final ClientEvaluationSubscriptions subscriptions;
    private final AtomicReference<CompletableFuture<Void>> actFuture = new AtomicReference<>();
    private Timeout noMessageTimeout;

    public ClientEvaluationStream(
            StreamObserver<EvaluationStreamClientMessage> output,
            Executor executor,
            ClientEvaluationBreaker evaluationBreaker)
    {
        this(new EvaluationStreamClientMessageWriter(output), executor, evaluationBreaker, NO_MESSAGE_TIMEOUT_MS);
    }

    public ClientEvaluationStream(
            EvaluationStreamClientMessageWriter output,
            Executor executor,
            ClientEvaluationBreaker evaluationBreaker,
            long noMessageTimeoutMs)
    {
        this.output = output;
        this.evaluationBreaker = evaluationBreaker;
        this.noMessageTimeoutMs = noMessageTimeoutMs;
        this.actor = new ActorRunner(this::act, executor);
        this.subscriptions = new ClientEvaluationSubscriptions(this::onNextUnassign);
        scheduleTimeout();
    }

    public void assign(ClientAssignReq req) {
        outboundAssign.enqueue(req);
        actor.schedule();
    }

    @Override
    public void onNext(EvaluationStreamServerMessage value) {
        inboundMessages.enqueue(value);
        actor.schedule();
    }

    private void onNextUnassign(EvaluationAssignmentKey key) {
        outboundUnassign.enqueue(key);
        actor.schedule();
    }

    @Override
    public void onError(Throwable t) {
        inboundDone.complete(Status.fromThrowable(t));
        actor.schedule();
    }

    @Override
    public void onCompleted() {
        inboundDone.complete(Status.OK);
        actor.schedule();
    }

    @Override
    public void close() {
        outboundDone.complete(Status.CANCELLED.withDescription("Closed on client side"));
        actor.schedule();
    }

    public boolean isDone() {
        return outboundDone.isDone() || inboundDone.isDone();
    }

    public int size() {
        return size + outboundAssign.size();
    }

    private void act() {
        var doneFuture = actFuture.getAndSet(null);
        try {
            processClose();
            processInbound();
            processOutbound();
        } catch (Throwable e) {
            logger.error("Evaluation stream failed", e);

            actor.schedule();
            outboundDone.complete(Status.fromThrowable(e));
        } finally {
            if (doneFuture != null) {
                doneFuture.complete(null);
            }
        }
    }

    public CompletableFuture<Void> awaitAct() {
        return actFuture.updateAndGet(prev -> {
            if (prev != null) {
                return prev;
            }
            return new CompletableFuture<>();
        });
    }

    private void processClose() {
        if (noMessageTimeout.isExpired()) {
            outboundDone.complete(Status.DEADLINE_EXCEEDED.withDescription("Cancel stuck evaluation"));
        }

        if (inboundDone.isDone() || outboundDone.isDone()) {
            subscriptions.close();

            var requests = outboundAssign.dequeueAll();
            for (var req : requests) {
                quiteClose(req.subscriber(), false);
            }
        }
    }

    private void processInbound() {
        if (inboundDone.isDone()) {
            outboundDone.complete(Status.OK);
            return;
        }

        processInboundMessages();
        size = subscriptions.size();
    }

    private void processInboundMessages() {
        var messages = inboundMessages.dequeueAll();
        if (messages.isEmpty()) {
            return;
        }

        noMessageTimeout.cancel();
        scheduleTimeout();
        for (var message : messages) {
            for (var evaluation : message.getEvaluationsList()) {
                if (!subscriptions.nextEvaluation(evaluation)) {
                    output.onNext(toUnassignEvaluation(evaluation.getAssignmentKey()));
                    continue;
                }

                if (evaluationBreaker.continueEvaluation(evaluation)) {
                    continue;
                }

                var key = evaluation.getAssignmentKey();
                if (subscriptions.remove(key)) {
                    output.onNext(toUnassignEvaluation(key));
                }
            }

            for (var error : message.getErrorsList()) {
                subscriptions.nextEvaluationError(error);
            }
        }
    }

    private void processOutbound() {
        processOutboundDone();
        processOutboundAssign();
        processOutboundUnassign();
        size = subscriptions.size();
        output.flush();
    }

    private void processOutboundDone() {
        if (output.isClosed()) {
            return;
        }

        if (!outboundDone.isDone()) {
            return;
        }

        var status = outboundDone.join();
        if (status.isOk()) {
            output.onCompleted();
        } else {
            output.onError(new StatusRuntimeExceptionNoStackTrace(status));
        }
    }

    private void processOutboundAssign() {
        size = subscriptions.size() + outboundAssign.size();
        var requests = outboundAssign.dequeueAll();
        int index = 0;
        boolean subscribed = false;
        try {
            for (; index < requests.size(); index++) {
                var req = requests.get(index);
                var key = subscriptions.subscribe(req.seqNo(), req.subscriber());
                subscribed = key != null;
                if (key != null) {
                    output.onNext(toAssignEvaluation(key, req.alert(), req.state()));
                }
            }
        } catch (Throwable e) {
            for (; index < requests.size(); index++) {
                quiteClose(requests.get(index).subscriber(), subscribed);
                subscribed = false;
            }

            throw new RuntimeException(e);
        }
    }

    private void quiteClose(Flow.Subscriber<?> subscriber, boolean subscribed) {
        if (!subscribed) {
            try {
                subscriber.onSubscribe(NoopSubscription.INSTANCE);
            } catch (Throwable e) {
                logger.warn("failed on subscribe", e);
            }
        }

        try {
            subscriber.onComplete();
        } catch (Throwable e) {
            logger.warn("failed on complete", e);
        }
    }

    private void processOutboundUnassign() {
        var keys = outboundUnassign.dequeueAll();
        for (var key : keys) {
            if (subscriptions.remove(key)) {
                output.onNext(toUnassignEvaluation(key));
            }
        }
    }

    private void scheduleTimeout() {
        noMessageTimeout = Async.runAfter(ignore -> actor.schedule(), noMessageTimeoutMs, TimeUnit.MILLISECONDS);
    }
}
