package ru.yandex.solomon.alert.cluster.server.grpc.evaluation;

import java.util.concurrent.Flow;
import java.util.concurrent.Flow.Subscriber;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;

import com.google.protobuf.TextFormat;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.alert.protobuf.EvaluationAssignmentKey;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.Evaluation;

/**
 * @author Vladimir Gordiychuk
 */
class ClientEvaluationSubscription implements Flow.Subscription {
    private static final Logger logger = LoggerFactory.getLogger(ClientEvaluationSubscription.class);

    private final EvaluationAssignmentKey key;
    private final Subscriber<Evaluation> subscriber;
    private final Consumer<EvaluationAssignmentKey> onCancel;
    private final AtomicBoolean done = new AtomicBoolean();

    ClientEvaluationSubscription(
            EvaluationAssignmentKey key,
            Subscriber<Evaluation> subscriber,
            Consumer<EvaluationAssignmentKey> onCancel)
    {
        this.key = key;
        this.subscriber = subscriber;
        this.onCancel = onCancel;
    }

    @Override
    public void request(long n) {
    }

    public boolean onSubscribe() {
        try {
            subscriber.onSubscribe(this);
            return true;
        } catch (Throwable e) {
            logger.error("{} onSubscribe failed", this, e);
            onError(e);
            return false;
        }
    }

    public boolean onNextEvaluation(Evaluation evaluation) {
        if (done.get()) {
            return false;
        }

        try {
            subscriber.onNext(evaluation);
            return true;
        } catch (Throwable e) {
            logger.error("{} onNext failed at evaluation {}", this, TextFormat.shortDebugString(evaluation), e);
            onError(e);
            return false;
        }
    }

    public boolean onError(Throwable e) {
        if (markDone()) {
            try {
                subscriber.onError(e);
            } catch (Throwable e2) {
                e2.addSuppressed(e);
                logger.error("{} onError failed", this, e2);
            }
            return true;
        }

        return false;
    }

    public boolean onError(Status status) {
        return onError(new StatusRuntimeExceptionNoStackTrace(status));
    }

    public void onComplete() {
        if (markDone()) {
            try {
                subscriber.onComplete();
            } catch (Throwable e) {
                logger.error("{} onComplete failed", this, e);
            }
        }
    }

    @Override
    public void cancel() {
        if (markDone()) {
            onCancel.accept(key);
        }
    }

    private boolean markDone() {
        return done.compareAndSet(false, true);
    }


    @Override
    public String toString() {
        return "ClientEvaluationSubscription{" +
                "key=EvaluationAssignmentKey{" + TextFormat.shortDebugString(key) + "}" +
                ", subscriber=" + subscriber +
                '}';
    }
}
