package ru.yandex.solomon.alert.cluster.server.grpc.evaluation;

import java.util.concurrent.Flow.Subscriber;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import io.grpc.Status;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.alert.protobuf.EvaluationAssignmentKey;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.Evaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.EvaluationError;
import ru.yandex.solomon.alert.protobuf.TAssignmentSeqNo;

import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toKey;
import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toStatus;

/**
 * @author Vladimir Gordiychuk
 */
public class ClientEvaluationSubscriptionGroup implements AutoCloseable {
    private final TAssignmentSeqNo groupId;
    private final Consumer<EvaluationAssignmentKey> onCancel;
    private final Long2ObjectMap<ClientEvaluationSubscription> subscriptionByAssignId = new Long2ObjectOpenHashMap<>();

    public ClientEvaluationSubscriptionGroup(TAssignmentSeqNo groupId, Consumer<EvaluationAssignmentKey> onCancel) {
        this.groupId = groupId;
        this.onCancel = onCancel;
    }

    @Nullable
    public EvaluationAssignmentKey subscribe(long assignId, Subscriber<Evaluation> subscriber) {
        var key = toKey(groupId, assignId);
        var result = new ClientEvaluationSubscription(key, subscriber, onCancel);
        var prev = subscriptionByAssignId.put(assignId, result);
        if (prev != null) {
            prev.onComplete();
        }

        if (result.onSubscribe()) {
            return key;
        }

        subscriptionByAssignId.remove(assignId);
        return null;
    }

    public boolean nextEvaluation(Evaluation evaluation) {
        var key = evaluation.getAssignmentKey();
        var subscription = subscriptionByAssignId.get(key.getAssignId());
        if (subscription == null) {
            return false;
        }

        if (subscription.onNextEvaluation(evaluation)) {
            return true;
        }

        subscriptionByAssignId.remove(key.getAssignId());
        return false;
    }

    public boolean nextEvaluationError(EvaluationError error) {
        var assignId = error.getAssignmentKey().getAssignId();

        Status status = toStatus(error.getStatus());
        if (assignId == 0) {
            var e = new StatusRuntimeExceptionNoStackTrace(status);
            for (var subscription : subscriptionByAssignId.values()) {
                subscription.onError(e);
            }
            subscriptionByAssignId.clear();
            return true;
        }

        var subscription = subscriptionByAssignId.remove(assignId);
        if (subscription == null) {
            return false;
        }

        return subscription.onError(status);
    }

    public boolean remove(EvaluationAssignmentKey key) {
        var assignId = key.getAssignId();
        if (assignId == 0) {
            for (var subscription : subscriptionByAssignId.values()) {
                subscription.onComplete();
            }
            subscriptionByAssignId.clear();
            return true;
        }

        var subscription = subscriptionByAssignId.remove(assignId);
        if (subscription != null) {
            subscription.onComplete();
            return true;
        }

        return false;
    }

    public int size() {
        return subscriptionByAssignId.size();
    }

    public boolean isEmpty() {
        return subscriptionByAssignId.isEmpty();
    }

    @Override
    public void close() {
        for (var subscription : subscriptionByAssignId.values()) {
            subscription.onComplete();
        }
        subscriptionByAssignId.clear();
    }
}
