package ru.yandex.solomon.alert.cluster.server.grpc.evaluation;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Flow.Subscriber;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import ru.yandex.solomon.alert.protobuf.EvaluationAssignmentKey;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.Evaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.EvaluationError;
import ru.yandex.solomon.alert.protobuf.TAssignmentSeqNo;

/**
 * @author Vladimir Gordiychuk
 */
public class ClientEvaluationSubscriptions implements AutoCloseable {

    private final Consumer<EvaluationAssignmentKey> onCancel;
    private long nextAssignId = 1;
    private final Map<TAssignmentSeqNo, ClientEvaluationSubscriptionGroup> subscriptionGroupById = new HashMap<>();
    private int size;

    public ClientEvaluationSubscriptions(Consumer<EvaluationAssignmentKey> onCancel) {
        this.onCancel = onCancel;
    }

    @Nullable
    public EvaluationAssignmentKey subscribe(TAssignmentSeqNo groupId, Subscriber<Evaluation> subscriber) {
        var group = getOrCreateGroup(groupId);
        var prevSize = group.size();
        var result = group.subscribe(nextAssignId++, subscriber);
        if (result == null && group.isEmpty()) {
            subscriptionGroupById.remove(groupId);
        }
        this.size += group.size() - prevSize;
        return result;
    }

    public boolean nextEvaluation(Evaluation evaluation) {
        var groupId = evaluation.getAssignmentKey().getAssignGroupId();
        var group = subscriptionGroupById.get(groupId);
        if (group == null) {
            return false;
        }

        var prevSize = group.size();
        try {
            if (group.nextEvaluation(evaluation)) {
                return true;
            }

            if (group.isEmpty()) {
                subscriptionGroupById.remove(groupId);
            }
        } finally {
            this.size += group.size() - prevSize;
        }
        return false;
    }

    public boolean nextEvaluationError(EvaluationError error) {
        var groupId = error.getAssignmentKey().getAssignGroupId();
        var group = subscriptionGroupById.get(groupId);
        if (group == null) {
            return false;
        }

        int prevSize = group.size();
        try {
            if (!group.nextEvaluationError(error)) {
                return false;
            }

            if (group.isEmpty()) {
                subscriptionGroupById.remove(groupId);
            }
        } finally {
            this.size += group.size() - prevSize;
        }
        return true;
    }

    public boolean remove(EvaluationAssignmentKey key) {
        var groupId = key.getAssignGroupId();
        var group = subscriptionGroupById.get(groupId);
        if (group == null) {
            return false;
        }

        int prevSize = group.size();
        try {
            if (!group.remove(key)) {
                return false;
            }

            if (group.isEmpty()) {
                subscriptionGroupById.remove(groupId);
            }
        } finally {
            this.size += group.size() - prevSize;
        }
        return true;
    }

    public boolean isEmpty() {
        return subscriptionGroupById.isEmpty();
    }

    public int size() {
        return size;
    }

    @Override
    public void close() {
        for (var subscription : subscriptionGroupById.values()) {
            subscription.close();
        }
        subscriptionGroupById.clear();
        size = 0;
    }

    private ClientEvaluationSubscriptionGroup getOrCreateGroup(TAssignmentSeqNo seqNo) {
        var result = subscriptionGroupById.get(seqNo);
        if (result == null) {
            result = new ClientEvaluationSubscriptionGroup(seqNo, onCancel);
            subscriptionGroupById.put(seqNo, result);
        }

        return result;
    }
}
