package ru.yandex.solomon.alert.cost;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;

import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPoint;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ThreadSafe
@ParametersAreNonnullByDefault
public class EstimationOptions {
    private static final int DEFAULT_APPROXIMATE_COUNT_METRICS_BY_SELECTOR = 3;
    private static final long DEFAULT_APPROXIMATE_POINT_STEP_MILLIS = TimeUnit.SECONDS.toMillis(15);
    private static final double DEFAULT_METRIC_LOAD_COST = 2d;
    private static final double DEFAULT_POINT_COST = MemoryCounter.objectSelfSizeLayout(AggrPoint.class);

    private static final EstimationOptions EMPTY = EstimationOptions.newBuilder().build();

    /**
     * This parameters allow to guess how much metrics will be return by user specified selector, when
     * other methods doesn't work, because for example metabase not available, older metrics was removed
     * but new not created yet, etc.
     */
    private final int approximateCountMetricsBySelector;

    /**
     * This parameters allow to guess how much points will be return by configured alert period, when
     * other methods doesn't work, because for example metabase/stockpile not available, during check
     * metric doesn't have point yet, etc.
     */
    private final long approximatePointStepMillis;

    /**
     * Estimate the cost of a metric load. The default value is 2.
     */
    private final double metricLoadCost;

    /**
     * Estimate the cost of a single point for time series. The default value is {@link AggrPoint}
     * object size. This parameter correlated with memory usage to evaluate alert rule.
     */
    private final double pointCost;

    private EstimationOptions(Builder builder) {
        this.approximateCountMetricsBySelector = builder.approximateCountMetricsBySelector;
        this.approximatePointStepMillis = builder.approximatePointStepMillis;
        this.metricLoadCost = builder.metricLoadCost;
        this.pointCost = builder.pointCost;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static EstimationOptions empty() {
        return EMPTY;
    }

    public int getApproximateCountMetricsBySelector() {
        return approximateCountMetricsBySelector;
    }

    public long getApproximatePointStepMillis() {
        return approximatePointStepMillis;
    }

    public double getMetricLoadCost() {
        return metricLoadCost;
    }

    public double getPointCost() {
        return pointCost;
    }

    @Override
    public String toString() {
        return "EstimationOptions{" +
                "approximateCountMetricsBySelector=" + approximateCountMetricsBySelector +
                ", approximatePointStepMillis=" + approximatePointStepMillis +
                ", metricLoadCost=" + metricLoadCost +
                ", pointCost=" + pointCost +
                '}';
    }

    @NotThreadSafe
    public static class Builder {
        private int approximateCountMetricsBySelector = DEFAULT_APPROXIMATE_COUNT_METRICS_BY_SELECTOR;
        private long approximatePointStepMillis = DEFAULT_APPROXIMATE_POINT_STEP_MILLIS;
        private double metricLoadCost = DEFAULT_METRIC_LOAD_COST;
        private double pointCost = DEFAULT_POINT_COST;

        private Builder() {
        }

        /**
         * @see EstimationOptions#approximateCountMetricsBySelector
         */
        public Builder setApproximateCountMetricsBySelector(int count) {
            this.approximateCountMetricsBySelector = count;
            return this;
        }

        /**
         * @see EstimationOptions#approximatePointStepMillis
         */
        public Builder setApproximatePointStepMillis(long stepMillis) {
            this.approximatePointStepMillis = stepMillis;
            return this;
        }

        /**
         * @see EstimationOptions#metricLoadCost
         */
        public Builder setMetricLoadCost(double cost) {
            this.metricLoadCost = cost;
            return this;
        }

        /**
         * @see EstimationOptions#pointCost
         */
        public Builder setPointCost(double cost) {
            this.pointCost = cost;
            return this;
        }

        public EstimationOptions build() {
            return new EstimationOptions(this);
        }
    }
}
