package ru.yandex.solomon.alert.dao;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateId;
import ru.yandex.solomon.alert.template.domain.AlertTemplateLastVersion;
import ru.yandex.solomon.ydb.page.TokenBasePage;

/**
 * @author Alexey Trushkin
 */
public class CachedAlertTemplateDao implements AlertTemplateDao {

    private final Map<AlertTemplateId, AlertTemplate> cache = new ConcurrentHashMap<>();
    private final AlertTemplateDao dao;

    public CachedAlertTemplateDao(AlertTemplateDao dao) {
        this.dao = dao;
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return CompletableFuture.failedFuture(new UnsupportedOperationException("Can't create schema"));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return CompletableFuture.failedFuture(new UnsupportedOperationException("Can't drop schema"));
    }

    @Override
    public CompletableFuture<Boolean> create(AlertTemplate alertTemplate) {
        return dao.create(alertTemplate)
                .thenApply(aBoolean -> {
                    if (Boolean.TRUE.equals(aBoolean)) {
                        cache.putIfAbsent(alertTemplate.getCompositeId(), alertTemplate);
                    }
                    return aBoolean;
                });
    }

    @Override
    public CompletableFuture<Optional<AlertTemplate>> findById(String id, String templateVersionTag) {
        var templateId = new AlertTemplateId(id, templateVersionTag);
        if (cache.isEmpty()) {
            return getAll()
                    .thenApply(ignored -> Optional.ofNullable(cache.get(templateId)));
        }
        AlertTemplate template = cache.get(templateId);
        if (template != null) {
            return CompletableFuture.completedFuture(Optional.of(template));
        }
        return dao.findById(id, templateVersionTag)
                .thenApply(alertTemplate -> {
                    if (alertTemplate.isEmpty()) {
                        return alertTemplate;
                    }
                    cache.putIfAbsent(templateId, alertTemplate.get());
                    return Optional.ofNullable(cache.get(templateId));
                });
    }

    @Override
    public CompletableFuture<List<AlertTemplate>> getAll() {
        return dao.getAll()
                .thenApply(alertTemplates -> {
                    for (AlertTemplate alertTemplate : alertTemplates) {
                        cache.putIfAbsent(alertTemplate.getCompositeId(), alertTemplate);
                    }
                    return new ArrayList<>(cache.values());
                });
    }

    @Override
    public CompletableFuture<List<AlertTemplate>> findVersions(List<AlertTemplateLastVersion> items) {
        return dao.findVersions(items);
    }

    @Override
    public CompletableFuture<TokenBasePage<AlertTemplate>> listLastTemplates(String serviceProviderId, String nameFilter, String labelsSelector, int pageSize, String pageToken, Predicate<AlertTemplate> skip) {
        return dao.listLastTemplates(serviceProviderId, nameFilter, labelsSelector, pageSize, pageToken, skip);
    }

    @Override
    public CompletableFuture<TokenBasePage<AlertTemplate>> listTemplateVersions(String templateId, int pageSize, String pageToken) {
        return dao.listTemplateVersions(templateId, pageSize, pageToken)
                .thenApply(page -> {
                    for (AlertTemplate alertTemplate : page.getItems()) {
                        cache.putIfAbsent(alertTemplate.getCompositeId(), alertTemplate);
                    }
                    return page;
                });
    }

    @Override
    public CompletableFuture<Boolean> publish(AlertTemplate newAlertTemplate, int version) {
        return dao.publish(newAlertTemplate, version)
                .thenApply(aBoolean -> {
                    if (Boolean.TRUE.equals(aBoolean)) {
                        cache.putIfAbsent(newAlertTemplate.getCompositeId(), newAlertTemplate);
                    }
                    return aBoolean;
                });
    }

    @VisibleForTesting
    public Map<AlertTemplateId, AlertTemplate> getCache() {
        return cache;
    }
}
