package ru.yandex.solomon.alert.dao;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;
import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.conf.ConfigNotInitialized;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Project;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class ProjectHolderImpl implements ProjectsHolder, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ProjectHolderImpl.class);
    private static final Interner<ProjectView> I = Interners.newWeakInterner();
    private final ProjectsDao dao;
    private final ExecutorService executor;
    private final ScheduledExecutorService timer;
    private volatile ConcurrentMap<String, ProjectView> projectById = new ConcurrentHashMap<>();
    private volatile boolean scheduled;
    private volatile boolean closed;
    private volatile Future scheduledFuture = completedFuture(null);
    private volatile CompletableFuture<Void> nextReload = new CompletableFuture<>();
    private final ActorWithFutureRunner actor;

    public ProjectHolderImpl(ProjectsDao dao, ExecutorService executor, ScheduledExecutorService timer) {
        this.dao = dao;
        this.timer = timer;
        this.executor = executor;
        this.actor = new ActorWithFutureRunner(this::act, executor);
        this.actor.schedule();
    }

    @Override
    public CompletableFuture<Void> reload() {
        var future = nextReload;
        actor.schedule();
        return future;
    }

    public CompletableFuture<Void> awaitNextReload() {
        return nextReload;
    }

    @Override
    public CompletableFuture<Boolean> hasProject(String projectId) {
        if (projectById.containsKey(projectId)) {
            return completedFuture(Boolean.TRUE);
        }

        return dao.findById(projectId).thenApply(project -> {
            if (project.isPresent()) {
                var prev = projectById.put(projectId, createView(project.get()));
                if (prev == null) {
                    // force reload because memory state obsolete
                    actor.schedule();
                }
            }
            return project.isPresent();
        });
    }

    @Override
    public Optional<ProjectView> getProjectView(String id) {
        return Optional.ofNullable(projectById.get(id));
    }

    private ProjectView createView(Project project) {
        var view = new ProjectView(project.getId(), project.getAbcService());
        return I.intern(view);
    }

    @Override
    public Set<String> getProjects() {
        return projectById.keySet();
    }

    private CompletableFuture<Void> act() {
        if (closed) {
            return completedFuture(null);
        }

        return CompletableFutures.safeCall(dao::findAllNames)
                .thenAccept(projects -> {
                    if (projects.isEmpty() && !projectById.isEmpty()) {
                        throw new IllegalStateException("It's not available at one time remove all projects");
                    }

                    projectById = projects.stream().collect(Collectors.toConcurrentMap(Project::getId, this::createView));
                })
                .handle((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed to fetch list projects", e);
                        var root = Throwables.getRootCause(e);
                        if (root instanceof ConfigNotInitialized) {
                            scheduleNextRefresh(10_000, 15_000);
                        } else {
                            scheduleNextRefresh(10_000, 30_000);
                        }
                    } else {
                        scheduleNextRefresh(60_000, 120_000);
                        var future = nextReload;
                        nextReload = new CompletableFuture<>();
                        future.completeAsync(() -> null, executor);
                    }
                    return null;
                });
    }

    private void scheduleNextRefresh(int minWait, int maxWait) {
        if (scheduled || closed) {
            return;
        }

        long delay = ThreadLocalRandom.current().nextLong(minWait, maxWait);
        scheduledFuture = timer.schedule(() -> {
            scheduled = false;
            actor.schedule();
        }, delay, TimeUnit.MILLISECONDS);
    }

    @Override
    public void close() {
        scheduledFuture.cancel(false);
        closed = true;
    }

    public static record ProjectView(String id, String abc) {
    }
}
