package ru.yandex.solomon.alert.dao;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.function.Consumer;

import com.github.luben.zstd.Zstd;
import com.google.protobuf.ByteString;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.CodedOutputStream;
import com.google.protobuf.ExtensionRegistryLite;
import io.grpc.Status;

import ru.yandex.solomon.alert.protobuf.TPersistAlertState;

/**
 * @author Vladimir Gordiychuk
 */
public class StatesCompressor {
    public static CompressResult compress(List<TPersistAlertState> states) throws IOException {
        int rawCapacity = states.stream()
                .mapToInt(CodedOutputStream::computeMessageSizeNoTag)
                .sum();

        ByteBuffer rawBuffer = ByteBuffer.allocateDirect(rawCapacity);
        CodedOutputStream out = CodedOutputStream.newInstance(rawBuffer);
        for (TPersistAlertState state : states) {
            out.writeMessageNoTag(state);
        }
        out.flush();
        rawBuffer.rewind();
        int rawSize = rawBuffer.remaining();
        ByteBuffer compressed = Zstd.compress(rawBuffer, 3);
        int compressedSize = compressed.remaining();
        return new CompressResult(rawSize, compressedSize, compressed);
    }

    public static void decompress(ByteString content, int rawBytesSize, Consumer<TPersistAlertState> consumer) {
        ByteBuffer buffer = ByteBuffer.allocateDirect(content.size());
        content.copyTo(buffer);
        buffer.position(0);
        decompress(buffer, rawBytesSize, consumer);
    }

    public static void decompress(ByteBuffer buffer, int rawBytesSize, Consumer<TPersistAlertState> consumer) {
        try {
            ByteBuffer decompressed = Zstd.decompress(buffer, rawBytesSize);
            CodedInputStream in = CodedInputStream.newInstance(decompressed);
            while (!in.isAtEnd()) {
                TPersistAlertState state = in.readMessage(TPersistAlertState.parser(), ExtensionRegistryLite.getEmptyRegistry());
                consumer.accept(state);
            }
        } catch (Throwable e) {
            throw Status.DATA_LOSS.withCause(e).asRuntimeException();
        }
    }

    public static final class CompressResult {
        private final int rawBytesSize;
        private final int compressedBytesSize;
        private final ByteBuffer compressed;

        CompressResult(int rawBytesSize, int compressedBytesSize, ByteBuffer compressed) {
            this.rawBytesSize = rawBytesSize;
            this.compressedBytesSize = compressedBytesSize;
            this.compressed = compressed;
        }

        public int getRawBytesSize() {
            return rawBytesSize;
        }

        public int getCompressedBytesSize() {
            return compressedBytesSize;
        }

        public ByteBuffer getCompressed() {
            return compressed;
        }
    }
}
