package ru.yandex.solomon.alert.dao;

import java.time.Instant;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.notification.channel.AlertApiKey;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.telegram.EventAppearance;

/**
 * @author alexlovkov
 **/
public class TelegramEventRecord {
    /* Generic part */
    private final String id;
    private final long createdAt;
    private final long messageId;
    private final String projectId;

    /* Alert message specific part */
    private final String alertId;
    private final String subAlertId;
    /**
     * when event was created, so we could get photo at this time
     */
    private final long evaluatedAt;
    private final EvaluationStatus.Code alertStatusCode;

    /* Mute message specific part */
    private final String muteId;

    private final TelegramEventContext context;

    private final EventAppearance eventAppearance;

    public static TelegramEventRecord makePartial(String uuid, long createdAt) {
        return new TelegramEventRecord(uuid, createdAt, 0, "",
                "", "", 0, EvaluationStatus.Code.NO_DATA,
                "",
                new TelegramEventContext(),
                EventAppearance.UNKNOWN);
    }

    public static TelegramEventRecord forAlert(String uuid, Instant createdAt, long messageId, Event event, EventAppearance eventAppearance) {
        return new TelegramEventRecord(uuid, createdAt.toEpochMilli(), messageId, event.getAlertApiKey().projectId(),
                event.getAlertApiKey().alertId(),
                event.getAlertApiKey().subAlertId(),
                event.getState().getLatestEval().toEpochMilli(),
                event.getState().getStatus().getCode(),
                "",
                new TelegramEventContext(),
                eventAppearance
        );
    }

    public static TelegramEventRecord forMuteCreated(String uuid, Instant createdAt, long messageId, String projectId, String muteId) {
        return new TelegramEventRecord(uuid, createdAt.toEpochMilli(), messageId, projectId,
                "", "", 0, EvaluationStatus.Code.NO_DATA,
                muteId,
                new TelegramEventContext(),
                EventAppearance.MUTE_SET
        );
    }

    public static TelegramEventRecord forMuteResolved(TelegramEventRecord created) {
        return new TelegramEventRecord(created.getId(), created.getEvaluatedAt(), created.getMessageId(), created.getProjectId(),
                "", "", 0, EvaluationStatus.Code.NO_DATA,
                created.getMuteId(),
                created.getContext(),
                EventAppearance.MUTE_REMOVED
        );
    }

    public static TelegramEventRecord forward(String newUuid, Instant createdAt, long messageId, TelegramEventRecord upstream) {
        return new TelegramEventRecord(newUuid, createdAt.toEpochMilli(), messageId, upstream.getProjectId(),
                upstream.getAlertId(), upstream.getSubAlertId(), upstream.getEvaluatedAt(), upstream.getAlertStatusCode(),
                upstream.getMuteId(),
                upstream.getContext(),
                EventAppearance.FORWARD
        );
    }

    public static TelegramEventRecord forPrompt(
            String uuid,
            Instant createdAt,
            long messageId,
            String origEventUuid,
            EventAppearance promptType)
    {
        return new TelegramEventRecord(uuid, createdAt.toEpochMilli(), messageId, "",
                "",
                "",
                0,
                EvaluationStatus.Code.NO_DATA,
                origEventUuid,
                new TelegramEventContext(),
                promptType
        );
    }

    public TelegramEventRecord(
        String uuid,
        long createdAt,
        long messageId,
        String projectId,
        String alertId,
        String subAlertId,
        long evaluatedAt,
        EvaluationStatus.Code alertStatusCode,
        String muteId,
        TelegramEventContext context,
        EventAppearance eventAppearance)
    {
        this.id = uuid;
        this.createdAt = createdAt;
        this.messageId = messageId;
        this.projectId = projectId;

        this.alertId = alertId;
        this.subAlertId = subAlertId;
        this.evaluatedAt = evaluatedAt;
        this.alertStatusCode = alertStatusCode;

        this.muteId = muteId;

        this.context = context;
        this.eventAppearance = eventAppearance;
    }

    public String getId() {
        return id;
    }

    public long getCreatedAt() {
        return createdAt;
    }

    public long getMessageId() {
        return messageId;
    }

    public String getProjectId() {
        return projectId;
    }

    public String getAlertId() {
        return alertId;
    }

    public String getSubAlertId() {
        return subAlertId;
    }

    public AlertApiKey getAlertApiKey() {
        return new AlertApiKey(projectId, alertId, subAlertId);
    }

    public long getEvaluatedAt() {
        return evaluatedAt;
    }

    public EvaluationStatus.Code getAlertStatusCode() {
        return alertStatusCode;
    }

    public String getMuteId() {
        return muteId;
    }

    public String getOrigEventUuid() {
        return muteId;
    }

    public TelegramEventContext getContext() {
        return context;
    }

    public EventAppearance getEventAppearance() {
        return eventAppearance;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        TelegramEventRecord that = (TelegramEventRecord) o;

        if (createdAt != that.createdAt) {
            return false;
        }
        if (messageId != that.messageId) {
            return false;
        }
        if (evaluatedAt != that.evaluatedAt) {
            return false;
        }
        if (!id.equals(that.id)) {
            return false;
        }
        if (!projectId.equals(that.projectId)) {
            return false;
        }
        if (!alertId.equals(that.alertId)) {
            return false;
        }
        if (!subAlertId.equals(that.subAlertId)) {
            return false;
        }
        if (alertStatusCode != that.alertStatusCode) {
            return false;
        }
        if (!muteId.equals(that.muteId)) {
            return false;
        }
        if (!context.equals(that.context)) {
            return false;
        }
        return eventAppearance == that.eventAppearance;
    }

    @Override
    public int hashCode() {
        int result = id.hashCode();
        result = 31 * result + (int) (createdAt ^ (createdAt >>> 32));
        result = 31 * result + (int) (messageId ^ (messageId >>> 32));
        result = 31 * result + projectId.hashCode();
        result = 31 * result + alertId.hashCode();
        result = 31 * result + subAlertId.hashCode();
        result = 31 * result + (int) (evaluatedAt ^ (evaluatedAt >>> 32));
        result = 31 * result + alertStatusCode.hashCode();
        result = 31 * result + muteId.hashCode();
        result = 31 * result + context.hashCode();
        result = 31 * result + eventAppearance.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "TelegramEventRecord{" +
                "id='" + id + '\'' +
                ", createdAt=" + createdAt +
                ", messageId=" + messageId +
                ", projectId='" + projectId + '\'' +
                ", alertId='" + alertId + '\'' +
                ", subAlertId='" + subAlertId + '\'' +
                ", evaluatedAt=" + evaluatedAt +
                ", alertStatusCode=" + alertStatusCode +
                ", muteId='" + muteId + '\'' +
                ", context=" + context +
                ", eventAppearance=" + eventAppearance +
                '}';
    }
}
