package ru.yandex.solomon.alert.dao.codec;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.AbstractAlertBuilder;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.domain.AlertState;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.domain.NoPointsPolicy;
import ru.yandex.solomon.alert.domain.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.template.AlertParameter;
import ru.yandex.solomon.alert.domain.threshold.Compare;
import ru.yandex.solomon.alert.domain.threshold.PredicateRule;
import ru.yandex.solomon.alert.domain.threshold.TargetStatus;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.domain.threshold.ThresholdType;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertCodec implements AbstractCodec<Alert, AlertRecord> {
    private static final String PERIOD_MILLIS = "period_millis";

    private static final String PROGRAM = "program";
    private static final String CHECK_EXPRESSION = "check_expression";

    private static final String SELECTORS = "selectors";
    private static final String THRESHOLD_TYPE = "threshold_type";
    private static final String COMPARISON = "comparison";
    private static final String THRESHOLD = "threshold";

    private static final String PREDICATE_RULES = "predicate_rules";
    private static final String TARGET_STATUS = "target_status";
    private static final String TRANSFORMATIONS = "transformations";

    private static final String CHANNELS = "channels";
    private static final String NOTIFY_ABOUT_STATUSES = "notify_about_statuses";
    private static final String REPEAT_NOTIFICATION_DELAY_MILLIS = "repeat_notification_delay_millis";

    private static final String TEMPLATE_ID = "templateId";
    private static final String TEMPLATE_VERSION_TAG = "templateVersionTag";
    private static final String SERVICE_PROVIDER = "serviceProvider";
    private static final String PARAMS = "params";
    private static final String THRESHOLDS = "thresholds";
    private static final String SERVICE_PROVIDER_ANNOTATIONS = "serviceProviderAnnotations";
    private static final String SEVERITY = "severity";
    private static final String ESCALATIONS = "escalations";
    private static final TypeReference<List<AlertParameter>> ALERT_PARAM_TYPE_REFERENCE = new TypeReference<>() {};

    private final ObjectMapper mapper;

    public AlertCodec(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    @Override
    public AlertRecord encode(Alert alert) {
        AlertRecord record = new AlertRecord();
        record.id = alert.getId();
        record.projectId = alert.getProjectId();
        record.folderId = alert.getFolderId();
        record.version = alert.getVersion();
        record.name = alert.getName();
        record.description = alert.getDescription();
        record.state = alert.getState().getNumber();
        record.createdBy = alert.getCreatedBy();
        record.createdAt = alert.getCreatedAt();
        record.updatedBy = alert.getUpdatedBy();
        record.updatedAt = alert.getUpdatedAt();
        record.groupByLabels = encodeJsonArray(alert.getGroupByLabels());
        record.notificationChannels = encodeJsonArray(new ArrayList<>(alert.getNotificationChannels().keySet()));
        record.type = alert.getAlertType().getNumber();
        record.config = encodeConfig(alert);
        record.annotations = encodeJsonMap(alert.getAnnotations());
        record.labels = encodeJsonMap(alert.getLabels());
        record.delaySeconds = alert.getDelaySeconds();
        record.notificationConfig = encodeNotificationConfig(alert.getNotificationChannels());
        record.resolvedEmptyPolicy = alert.getResolvedEmptyPolicy().name();
        record.noPointsPolicy = alert.getNoPointsPolicy().name();
        return record;
    }

    public String encodeNotificationConfig(Map<String, ChannelConfig> notificationRules) {
        try {
            ObjectNode root = mapper.createObjectNode();
            ObjectNode channels = root.putObject(CHANNELS);

            for (var rule : notificationRules.entrySet()) {
                ObjectNode ruleJson = channels.putObject(rule.getKey());
                encodeChannelConfig(ruleJson, rule.getValue());
            }

            return mapper.writeValueAsString(root);
        } catch (JsonProcessingException e) {
            throw Throwables.propagate(e);
        }
    }

    private void encodeChannelConfig(ObjectNode ruleJson, ChannelConfig value) {
        if (value == ChannelConfig.EMPTY) {
            return; // encode as empty object
        }

        ArrayNode statuses = ruleJson.putArray(NOTIFY_ABOUT_STATUSES);
        for (var code : value.getNotifyAboutStatusesUnchecked()) {
            statuses.add(code.name());
        }
        ruleJson.put(REPEAT_NOTIFICATION_DELAY_MILLIS, value.getRepeatNotificationDelayUnchecked().toMillis());
    }

    public String encodeJsonMap(Map<String, String> map) {
        try {
            return mapper.writeValueAsString(map);
        } catch (JsonProcessingException e) {
            throw Throwables.propagate(e);
        }
    }

    public String encodeJsonArray(Collection<String> list) {
        try {
            ArrayNode root = mapper.createArrayNode();
            for (String value : list) {
                root.add(value);
            }

            return mapper.writeValueAsString(root);
        } catch (JsonProcessingException e) {
            throw Throwables.propagate(e);
        }
    }

    public String encodeConfig(Alert alert) {
        try {
            switch (alert.getAlertType()) {
                case EXPRESSION:
                    return encodeExpressionConfig((ExpressionAlert) alert);
                case THRESHOLD:
                    return encodeThresholdConfig((ThresholdAlert) alert);
                case FROM_TEMPLATE:
                    return encodeFromTemplateConfig((AlertFromTemplatePersistent) alert);
                default:
                    throw new UnsupportedOperationException("Unsupported alert type: " + alert.getAlertType());
            }
        } catch (JsonProcessingException e) {
            throw Throwables.propagate(e);
        }
    }

    private String encodeFromTemplateConfig(AlertFromTemplatePersistent alert) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(TEMPLATE_ID, alert.getTemplateId())
                .put(TEMPLATE_VERSION_TAG, alert.getTemplateVersionTag())
                .put(PARAMS, mapper.writeValueAsString(alert.getParameters()))
                .put(THRESHOLDS, mapper.writeValueAsString(alert.getThresholds()))
                .put(SERVICE_PROVIDER, alert.getServiceProvider())
                .put(PERIOD_MILLIS, alert.getPeriod().toMillis())
                .put(SERVICE_PROVIDER_ANNOTATIONS, encodeJsonMap(alert.getServiceProviderAnnotations()))
                .put(ESCALATIONS, encodeJsonArray(alert.getEscalations()));
        return mapper.writeValueAsString(root);
    }

    private String encodeExpressionConfig(ExpressionAlert alert) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(PROGRAM, alert.getProgram())
                .put(CHECK_EXPRESSION, alert.getCheckExpression())
                .put(PERIOD_MILLIS, alert.getPeriod().toMillis())
                .put(SEVERITY, alert.getSeverity().name())
                .put(SERVICE_PROVIDER_ANNOTATIONS, encodeJsonMap(alert.getServiceProviderAnnotations()))
                .put(ESCALATIONS, encodeJsonArray(alert.getEscalations()));

        return mapper.writeValueAsString(root);
    }

    private ObjectNode encodePredicateRule(PredicateRule rule) {
        return mapper.createObjectNode()
                .put(THRESHOLD_TYPE, rule.getThresholdType().name())
                .put(THRESHOLD, rule.getThreshold())
                .put(COMPARISON, rule.getComparison().name())
                .put(TARGET_STATUS, rule.getTargetStatus().name());
    }

    private String encodeThresholdConfig(ThresholdAlert alert) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(SELECTORS, Selectors.format(alert.getSelectors()))
                .put(PERIOD_MILLIS, alert.getPeriod().toMillis())
                .put(SEVERITY, alert.getSeverity().name())
                .put(SERVICE_PROVIDER_ANNOTATIONS, encodeJsonMap(alert.getServiceProviderAnnotations()))
                .put(ESCALATIONS, encodeJsonArray(alert.getEscalations()));

        PredicateRule predicateRule = alert.getPredicateRule();

        root.put(THRESHOLD_TYPE, predicateRule.getThresholdType().name())
            .put(THRESHOLD, predicateRule.getThreshold())
            .put(COMPARISON, predicateRule.getComparison().name());

        root.put(TRANSFORMATIONS, alert.getTransformations());

        List<ObjectNode> rules = alert.getPredicateRules().stream().map(this::encodePredicateRule).collect(Collectors.toList());
        root.putArray(PREDICATE_RULES).addAll(rules);

        return mapper.writeValueAsString(root);
    }

    @Override
    public Alert decode(AlertRecord record) {
        var alertBuilder = decodeConfig(record)
                .setId(record.id)
                .setProjectId(record.projectId)
                .setFolderId(record.folderId)
                .setVersion(record.version)
                .setName(record.name)
                .setDescription(record.description)
                .setState(AlertState.forNumber(record.state))
                .setCreatedBy(record.createdBy)
                .setCreatedAt(record.createdAt)
                .setUpdatedBy(record.updatedBy)
                .setUpdatedAt(record.updatedAt)
                .setGroupByLabels(decodeJsonArray(record.groupByLabels))
                .setAnnotations(decodeJsonMap(record.annotations))
                .setLabels(decodeJsonMap(record.labels))
                .setDelaySeconds(record.delaySeconds);

        if (!Strings.isNullOrEmpty(record.resolvedEmptyPolicy)) {
            alertBuilder.setResolvedEmptyPolicy(ResolvedEmptyPolicy.valueOf(record.resolvedEmptyPolicy));
        }
        if (!Strings.isNullOrEmpty(record.noPointsPolicy)) {
            alertBuilder.setNoPointsPolicy(NoPointsPolicy.valueOf(record.noPointsPolicy));
        }

        if (record.notificationConfig.isEmpty()) {
            alertBuilder.setNotificationChannels(decodeJsonArray(record.notificationChannels));
        } else {
            alertBuilder.setNotificationChannels(decodeNotificationConfig(record.notificationConfig));
        }

        return alertBuilder.build();
    }

    public Map<String, ChannelConfig> decodeNotificationConfig(String notificationConfig) {
        try {
            JsonNode channelsNode = mapper.readTree(notificationConfig).get(CHANNELS);

            Preconditions.checkState(channelsNode.isObject());
            ObjectNode channels = (ObjectNode) channelsNode;
            Map<String, ChannelConfig> result = new HashMap<>(channelsNode.size());

            channels.fields().forEachRemaining(entry -> result.put(entry.getKey(), parseChannelConfig(entry.getValue())));

            return result;
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    private ChannelConfig parseChannelConfig(JsonNode value) {
        Preconditions.checkState(value.isObject());

        ObjectNode channelConfig = (ObjectNode) value;

        if (channelConfig.size() == 0) {
            return ChannelConfig.EMPTY;
        }

        JsonNode statusesNode = channelConfig.get(NOTIFY_ABOUT_STATUSES);
        Preconditions.checkArgument(statusesNode.isArray());
        ArrayNode statuses = (ArrayNode) statusesNode;

        Set<EvaluationStatus.Code> notifyAbout = new HashSet<>(statuses.size());
        statuses.forEach(elem -> notifyAbout.add(EvaluationStatus.Code.parse(elem.textValue())));

        JsonNode repeatDelayNode = channelConfig.get(REPEAT_NOTIFICATION_DELAY_MILLIS);
        Duration repeatDelay = Duration.ofMillis(repeatDelayNode.asLong());

        return new ChannelConfig(notifyAbout, repeatDelay);
    }

    public Map<String, String> decodeJsonMap(String str) {
        if (str.isEmpty()) {
            return Collections.emptyMap();
        }

        try {
            return mapper.readValue(str, new TypeReference<Map<String, String>>(){});
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    public List<String> decodeJsonArray(String str) {
        try {
            JsonNode root = mapper.readTree(str);
            Preconditions.checkState(root.isArray());
            List<String> result = new ArrayList<>(root.size());
            for (JsonNode item : root) {
                result.add(item.asText());
            }

            return result;
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    private AbstractAlertBuilder decodeConfig(AlertRecord record) {
        AlertType type = AlertType.forNumber(record.type);
        return decodeConfig(type, record.config);
    }

    public AbstractAlertBuilder decodeConfig(AlertType type, String config) {
        try {
            switch (type) {
                case EXPRESSION:
                    return decodeExpressionConfig(config);
                case THRESHOLD:
                    return decodeThresholdConfig(config);
                case FROM_TEMPLATE:
                    return decodeFromTemplateConfig(config);
                default:
                    throw new UnsupportedOperationException("Not supported alert type: " + type);
            }
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    private AlertFromTemplatePersistent.Builder decodeFromTemplateConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        return AlertFromTemplatePersistent.newBuilder()
                .setTemplateId(root.get(TEMPLATE_ID).asText())
                .setTemplateVersionTag(root.get(TEMPLATE_VERSION_TAG).asText())
                .setServiceProvider(root.get(SERVICE_PROVIDER) == null ? "" : root.get(SERVICE_PROVIDER).asText())
                .setParameters(mapper.readValue(root.get(PARAMS).asText(), ALERT_PARAM_TYPE_REFERENCE))
                .setThresholds(mapper.readValue(root.get(THRESHOLDS).asText(), ALERT_PARAM_TYPE_REFERENCE))
                .setPeriod(Duration.ofMillis(root.get(PERIOD_MILLIS).asLong()))
                .setServiceProviderAnnotations(root.get(SERVICE_PROVIDER_ANNOTATIONS) == null ? Map.of() : decodeJsonMap(root.get(SERVICE_PROVIDER_ANNOTATIONS).asText()))
                .setEscalations(root.get(ESCALATIONS) == null ? Set.of() : new HashSet<>(decodeJsonArray(root.get(ESCALATIONS).asText())));
    }

    private ExpressionAlert.Builder decodeExpressionConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        return ExpressionAlert.newBuilder()
                .setProgram(root.get(PROGRAM).asText())
                .setCheckExpression(root.get(CHECK_EXPRESSION).asText())
                .setSeverity(getSeverity(root))
                .setServiceProviderAnnotations(root.get(SERVICE_PROVIDER_ANNOTATIONS) == null ? Map.of() : decodeJsonMap(root.get(SERVICE_PROVIDER_ANNOTATIONS).asText()))
                .setEscalations(root.get(ESCALATIONS) == null ? Set.of() : new HashSet<>(decodeJsonArray(root.get(ESCALATIONS).asText())))
                .setPeriod(Duration.ofMillis(root.get(PERIOD_MILLIS).asLong()));
    }

    private AlertSeverity getSeverity(JsonNode root) {
        return root.get(SEVERITY) == null
                ? AlertSeverity.UNKNOWN
                : AlertSeverity.valueOf(root.get(SEVERITY).asText());
    }

    private PredicateRule decodePredicateRule(JsonNode item) {
        return PredicateRule.onThreshold(item.get(THRESHOLD).asDouble())
                .withThresholdType(ThresholdType.valueOf(item.get(THRESHOLD_TYPE).asText()))
                .withComparison(Compare.valueOf(item.get(COMPARISON).asText()))
                .withTargetStatus(TargetStatus.valueOf(item.get(TARGET_STATUS).asText()));
    }

    private ThresholdAlert.Builder decodeThresholdConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        ThresholdAlert.Builder builder = ThresholdAlert.newBuilder()
                .setSelectors(Selectors.parse(root.get(SELECTORS).asText()))
                .setPeriod(Duration.ofMillis(root.get(PERIOD_MILLIS).asLong()))
                .setSeverity(getSeverity(root))
                .setEscalations(root.get(ESCALATIONS) == null ? Set.of() : new HashSet<>(decodeJsonArray(root.get(ESCALATIONS).asText())))
                .setServiceProviderAnnotations(root.get(SERVICE_PROVIDER_ANNOTATIONS) == null ? Map.of() : decodeJsonMap(root.get(SERVICE_PROVIDER_ANNOTATIONS).asText()));

        builder.setPredicateRule(PredicateRule
                .onThreshold(root.get(THRESHOLD).asDouble())
                .withThresholdType(ThresholdType.valueOf(root.get(THRESHOLD_TYPE).asText()))
                .withComparison(Compare.valueOf(root.get(COMPARISON).asText())));

        if (root.has(TRANSFORMATIONS)) {
            builder.setTransformations(root.get(TRANSFORMATIONS).asText());
        }

        if (root.has(PREDICATE_RULES)) {
            JsonNode predicateRules = root.get(PREDICATE_RULES);
            Preconditions.checkState(predicateRules.isArray());
            List<PredicateRule> rules = new ArrayList<>(predicateRules.size());
            for (JsonNode item : predicateRules) {
                rules.add(decodePredicateRule(item));
            }
            builder.setPredicateRules(rules.stream());
        }

        return builder;
    }

}
