package ru.yandex.solomon.alert.dao.codec;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.mute.domain.MuteType;
import ru.yandex.solomon.alert.mute.domain.SelectorsMute;
import ru.yandex.solomon.labels.query.SelectorsFormat;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class MuteCodec implements AbstractCodec<Mute, MuteRecord> {
    private static final String ALERT_SELECTOR = "alertSelector";
    private static final String LABEL_SELECTORS = "labelSelectors";

    private final ObjectMapper mapper;

    public MuteCodec(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    private String encodeConfig(Mute entity) {
        try {
            return switch (entity.getType()) {
                case BY_SELECTORS -> encodeSelectorsConfig((SelectorsMute) entity);
            };
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private Mute.MuteBuilder<?, ?> decodeConfig(MuteRecord r) {
        MuteType dt = MuteType.byNumber(r.type);
        try {
            return switch (dt) {
                case BY_SELECTORS -> decodeSelectorsConfig(r.config);
            };
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private String encodeSelectorsConfig(SelectorsMute e) throws JsonProcessingException {
        var root = mapper.createObjectNode()
                .put(ALERT_SELECTOR, SelectorsFormat.format(e.getAlertSelector()))
                .put(LABEL_SELECTORS, SelectorsFormat.format(e.getLabelSelectors()))
                ;
        return mapper.writeValueAsString(root);
    }

    private SelectorsMute.Builder decodeSelectorsConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        return SelectorsMute.newBuilder()
                .setAlertSelector(SelectorsFormat.parseSelector(root.get(ALERT_SELECTOR).textValue()))
                .setLabelSelectors(SelectorsFormat.parse(root.get(LABEL_SELECTORS).textValue()))
                ;
    }

    @Override
    public MuteRecord encode(Mute e) {
        var r = new MuteRecord();
        r.id = e.getId();
        r.projectId = e.getProjectId();
        r.folderId = e.getFolderId();
        r.name = e.getName();
        r.description = e.getDescription();
        r.ticketId = e.getTicketId();
        r.from = e.getFrom().toEpochMilli();
        r.to = e.getTo().toEpochMilli();
        r.ttlBase = e.getTtlBase().toEpochMilli();
        r.version = e.getVersion();
        r.createdBy = e.getCreatedBy();
        r.updatedBy = e.getUpdatedBy();
        r.createdAt = e.getCreatedAt();
        r.updatedAt = e.getUpdatedAt();

        r.type = e.getType().getNumber();
        r.config = encodeConfig(e);

        return r;
    }

    @Override
    public Mute decode(MuteRecord r) {
        var builder = decodeConfig(r)
                .setId(r.id)
                .setProjectId(r.projectId)
                .setFolderId(r.folderId)
                .setName(r.name)
                .setDescription(r.description)
                .setTicketId(r.ticketId)
                .setFrom(Instant.ofEpochMilli(r.from))
                .setTo(Instant.ofEpochMilli(r.to))
                .setVersion(r.version)
                .setCreatedBy(r.createdBy)
                .setUpdatedBy(r.updatedBy)
                .setCreatedAt(Instant.ofEpochMilli(r.createdAt))
                .setUpdatedAt(Instant.ofEpochMilli(r.updatedAt));

        if (r.ttlBase == 0) {
            builder.setTtlBase(Instant.ofEpochMilli(r.to));
        } else {
            builder.setTtlBase(Instant.ofEpochMilli(r.ttlBase));
        }

        return builder.build();
    }
}
