package ru.yandex.solomon.alert.dao.codec;

import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Throwables;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.alert.notification.domain.PhoneNotification;
import ru.yandex.solomon.alert.notification.domain.email.CloudEmailNotification;
import ru.yandex.solomon.alert.notification.domain.email.DatalensEmailNotification;
import ru.yandex.solomon.alert.notification.domain.email.EmailNotification;
import ru.yandex.solomon.alert.notification.domain.juggler.JugglerNotification;
import ru.yandex.solomon.alert.notification.domain.push.CloudPushNotification;
import ru.yandex.solomon.alert.notification.domain.sms.CloudSmsNotification;
import ru.yandex.solomon.alert.notification.domain.sms.SmsNotification;
import ru.yandex.solomon.alert.notification.domain.telegram.TelegramNotification;
import ru.yandex.solomon.alert.notification.domain.webhook.WebhookNotification;
import ru.yandex.solomon.alert.notification.domain.yachats.YaChatsNotification;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
public class NotificationCodec implements AbstractCodec<Notification, NotificationRecord> {
    private static final String RECIPIENTS = "recipients";
    private static final String SUBJECT = "subject";
    private static final String CONTENT = "content";

    private static final String URL = "url";
    private static final String TEMPLATE = "template";
    private static final String HEADERS = "headers";

    private static final String PHONE = "phone";
    private static final String LOGIN = "login";

    private static final String TELEGRAM_CHAT_ID = "chatId";
    private static final String TELEGRAM_LOGIN = "login";
    private static final String SEND_SCREENSHOT = "sendScreenshot";

    private static final String YACHAT_LOGIN = "login";
    private static final String YACHAT_GROUP_ID = "groupId";

    private static final String HOST = "host";
    private static final String SERVICE = "service";
    private static final String INSTANCE = "instance";
    private static final String DESCRIPTION = "description";
    private static final String TAGS = "tags";

    private static final String ABC_SERVICE = "abcService";
    private static final String DUTY_SLUG = "dutySlug";
    private static final String TYPE = "type";

    private final ObjectMapper mapper;

    public NotificationCodec(ObjectMapper mapper) {
        this.mapper = mapper;
    }

    private EmailNotification.Builder decodeEmailConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        List<String> recipients = new ArrayList<>();
        for (JsonNode node : root.get(RECIPIENTS)) {
            recipients.add(node.textValue());
        }

        return EmailNotification.newBuilder()
                .setRecipients(recipients)
                .setSubjectTemplate(root.get(SUBJECT).asText())
                .setContentTemplate(root.get(CONTENT).asText());
    }

    private CloudEmailNotification.Builder decodeCloudEmailConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        List<String> recipients = new ArrayList<>();
        for (JsonNode node : root.get(RECIPIENTS)) {
            recipients.add(node.textValue());
        }

        return CloudEmailNotification.newBuilder()
                .setRecipients(recipients);
    }

    private CloudSmsNotification.Builder decodeCloudSmsConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        List<String> recipients = new ArrayList<>();
        for (JsonNode node : root.get(RECIPIENTS)) {
            recipients.add(node.textValue());
        }

        return CloudSmsNotification.newBuilder()
                .setRecipients(recipients);
    }

    private CloudPushNotification.Builder decodeCloudPushConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        List<String> recipients = new ArrayList<>();
        for (JsonNode node : root.get(RECIPIENTS)) {
            recipients.add(node.textValue());
        }

        return CloudPushNotification.newBuilder()
                .setRecipients(recipients);
    }

    private WebhookNotification.Builder decodeWebhookConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        JsonNode headers = root.get(HEADERS);
        Map<String, String> decodedHeaders = new LinkedHashMap<>();
        headers.iterator();
        for (Iterator<Map.Entry<String, JsonNode>> it = headers.fields(); it.hasNext(); ) {
            Map.Entry<String, JsonNode> entry = it.next();
            decodedHeaders.put(entry.getKey(), entry.getValue().asText());
        }

        return WebhookNotification.newBuilder()
                .setUrl(root.get(URL).asText())
                .setTemplate(root.get(TEMPLATE).asText())
                .setHeaders(decodedHeaders);
    }

    private PhoneNotification.Builder decodePhoneConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        var login = root.get(LOGIN).asText();
        var abc = root.get(ABC_SERVICE).asText();
        var dutySlug = root.get(DUTY_SLUG).asText();
        var type = root.get(TYPE).asText();

        if (type.equals("login")) {
            return PhoneNotification.newBuilder()
                    .setLogin(login)
                    .setDuty(PhoneNotification.AbcDuty.EMPTY);
        } else {
            return PhoneNotification.newBuilder()
                    .setLogin(login)
                    .setDuty(new PhoneNotification.AbcDuty(abc, dutySlug));
        }
    }

    private JugglerNotification.Builder decodeJugglerConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        List<String> tags = new ArrayList<>();
        for (JsonNode node : root.get(TAGS)) {
            tags.add(node.textValue());
        }

        var builder = JugglerNotification.newBuilder()
                .setHost(root.get(HOST).asText())
                .setService(root.get(SERVICE).asText())
                .setJugglerDescription(root.get(DESCRIPTION).asText())
                .setTags(tags);

        JsonNode instance = root.get(INSTANCE);
        if (instance != null && !instance.asText().isEmpty()) {
            builder.setInstance(instance.asText());
        }

        return builder;
    }

    private SmsNotification.Builder decodeSmsConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);

        SmsNotification.Builder builder = SmsNotification.newBuilder()
                .setTextTemplate(root.get(TEMPLATE).asText());

        JsonNode phone = root.get(PHONE);
        if (phone != null && StringUtils.isNotEmpty(phone.asText())) {
            return builder.setPhone(phone.asText());
        }

        JsonNode login = root.get(LOGIN);
        if (login != null && StringUtils.isNotEmpty(login.asText())) {
            return builder.setLogin(login.asText());
        }

        return builder;
    }

    private TelegramNotification.Builder decodeTelegramConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);

        TelegramNotification.Builder builder = TelegramNotification.newBuilder()
                .setTextTemplate(root.get(TEMPLATE).asText());

        JsonNode screenshotNode = root.get(SEND_SCREENSHOT);
        if (screenshotNode != null) {
            builder.setSendScreenshot(screenshotNode.asBoolean());
        }

        JsonNode chatId = root.get(TELEGRAM_CHAT_ID);
        if (chatId != null) {
            builder.setChatId(chatId.asLong());
        }

        JsonNode login = root.get(TELEGRAM_LOGIN);
        if (login != null && StringUtils.isNotEmpty(login.asText())) {
            builder.setLogin(login.asText());
        }
        return builder;
    }

    private YaChatsNotification.Builder decodeYaChatsConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);

        YaChatsNotification.Builder builder = YaChatsNotification.newBuilder()
                .setTextTemplate(root.get(TEMPLATE).textValue());

        JsonNode login = root.get(YACHAT_LOGIN);
        if (login != null && StringUtils.isNotEmpty(login.textValue())) {
            builder.setLogin(login.textValue());
        }

        JsonNode groupId = root.get(YACHAT_GROUP_ID);
        if (groupId != null && StringUtils.isNotEmpty(groupId.textValue())) {
            builder.setGroupId(groupId.textValue());
        }

        return builder;
    }

    private DatalensEmailNotification.Builder decodeDatalensEmailConfig(String config) throws IOException {
        JsonNode root = mapper.readTree(config);
        List<String> recipients = new ArrayList<>();
        for (JsonNode node : root.get(RECIPIENTS)) {
            recipients.add(node.textValue());
        }

        return DatalensEmailNotification.newBuilder()
                .setRecipients(recipients);
    }

    public String encodeConfig(Notification channel) {
        try {
            return switch (channel.getType()) {
                case JUGGLER -> encodeJugglerConfig((JugglerNotification) channel);
                case EMAIL -> encodeEmailConfig((EmailNotification) channel);
                case WEBHOOK -> encodeWebhookConfig((WebhookNotification) channel);
                case PHONE_CALL -> encodePhoneConfig((PhoneNotification) channel);
                case SMS -> encodeSmsConfig((SmsNotification) channel);
                case TELEGRAM -> encodeTelegramConfig((TelegramNotification) channel);
                case CLOUD_EMAIL -> encodeCloudEmailConfig((CloudEmailNotification) channel);
                case CLOUD_SMS -> encodeCloudSmsConfig((CloudSmsNotification) channel);
                case YA_CHATS -> encodeYaChatsConfig((YaChatsNotification) channel);
                case DATALENS_EMAIL -> encodeDatalensEmailConfig((DatalensEmailNotification) channel);
                case CLOUD_PUSH -> encodeCloudPushConfig((CloudPushNotification) channel);
                default -> throw new UnsupportedOperationException("Unsupported notification type: " + channel.getType());
            };
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private String encodeJugglerConfig(JugglerNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(HOST, channel.getHost())
                .put(SERVICE, channel.getService())
                .put(INSTANCE, channel.getInstance())
                .put(DESCRIPTION, channel.getJugglerDescription());

        ArrayNode tags = root.putArray(TAGS);
        for (String tag : channel.getTags()) {
            tags.add(tag);
        }

        return mapper.writeValueAsString(root);
    }

    private String encodeEmailConfig(EmailNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(SUBJECT, channel.getSubjectTemplate())
                .put(CONTENT, channel.getContentTemplate());

        ArrayNode recipients = root.putArray(RECIPIENTS);
        for (String recipient : channel.getRecipients()) {
            recipients.add(recipient);
        }

        return mapper.writeValueAsString(root);
    }

    private String encodeCloudEmailConfig(CloudEmailNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode();

        ArrayNode recipients = root.putArray(RECIPIENTS);
        for (String recipient : channel.getRecipients()) {
            recipients.add(recipient);
        }

        return mapper.writeValueAsString(root);
    }

    private String encodeCloudSmsConfig(CloudSmsNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode();

        ArrayNode recipients = root.putArray(RECIPIENTS);
        for (String recipient : channel.getRecipients()) {
            recipients.add(recipient);
        }

        return mapper.writeValueAsString(root);
    }

    private String encodeCloudPushConfig(CloudPushNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode();

        ArrayNode recipients = root.putArray(RECIPIENTS);
        for (String recipient : channel.getRecipients()) {
            recipients.add(recipient);
        }

        return mapper.writeValueAsString(root);
    }

    private String encodeWebhookConfig(WebhookNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(URL, channel.getUrl())
                .put(TEMPLATE, channel.getTemplate());

        ObjectNode headers = root.putObject(HEADERS);
        for (Map.Entry<String, String> entry : channel.getHeaders().entrySet()) {
            headers.put(entry.getKey(), entry.getValue());
        }
        return mapper.writeValueAsString(root);
    }

    private String encodePhoneConfig(PhoneNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(TYPE, channel.isLogin() ? "login" : "duty")
                .put(LOGIN, channel.getLogin())
                .put(ABC_SERVICE, channel.getDuty().abcService())
                .put(DUTY_SLUG, channel.getDuty().dutySlug());
        return mapper.writeValueAsString(root);
    }

    private String encodeSmsConfig(SmsNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(PHONE, channel.getPhone())
                .put(LOGIN, channel.getLogin())
                .put(TEMPLATE, channel.getTextTemplate());
        return mapper.writeValueAsString(root);
    }

    private String encodeTelegramConfig(TelegramNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(TELEGRAM_CHAT_ID, channel.getChatId())
                .put(TELEGRAM_LOGIN, channel.getLogin())
                .put(TEMPLATE, channel.getTextTemplate())
                .put(SEND_SCREENSHOT, channel.isSendScreenshot());
        return mapper.writeValueAsString(root);
    }

    private String encodeYaChatsConfig(YaChatsNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode()
                .put(YACHAT_LOGIN, channel.getLogin())
                .put(YACHAT_GROUP_ID, channel.getGroupId())
                .put(TEMPLATE, channel.getTextTemplate());
        return mapper.writeValueAsString(root);
    }

    private String encodeDatalensEmailConfig(DatalensEmailNotification channel) throws JsonProcessingException {
        ObjectNode root = mapper.createObjectNode();

        ArrayNode recipients = root.putArray(RECIPIENTS);
        for (String recipient : channel.getRecipients()) {
            recipients.add(recipient);
        }

        return mapper.writeValueAsString(root);
    }


    @Override
    public NotificationRecord encode(Notification channel) {
        NotificationRecord record = new NotificationRecord();
        record.id = channel.getId();
        record.projectId = channel.getProjectId();
        record.folderId = channel.getFolderId();
        record.name = channel.getName();
        record.description = channel.getDescription();
        record.subscribeOn = encodeSubscribeOn(channel);
        record.repeatNotifyDelay = channel.getRepeatNotifyDelay().toMillis();
        record.createdBy = channel.getCreatedBy();
        record.createdAt = channel.getCreatedAt();
        record.updatedBy = channel.getUpdatedBy();
        record.updatedAt = channel.getUpdatedAt();
        record.version = channel.getVersion();
        record.type = channel.getType().getNumber();
        record.config = encodeConfig(channel);
        record.labels = encodeJsonMap(channel.getLabels());
        record.defaultForProject = channel.isDefaultForProject();
        record.defaultForSeverity = channel.getDefaultForSeverity().stream().map(Enum::name).collect(Collectors.toSet());
        return record;
    }

    public int encodeSubscribeOn(Notification channel) {
        return channel.getNotifyAboutStatus()
                .stream()
                .mapToInt(value -> 1 << value.getNumber())
                .reduce(0, (left, right) -> left | right);
    }

    @Override
    public Notification decode(NotificationRecord record) {
        return decodeConfig(record)
                .setId(record.id)
                .setProjectId(record.projectId)
                .setFolderId(record.folderId)
                .setName(record.name)
                .setDescription(record.description)
                .setCreatedBy(Nullables.orEmpty(record.createdBy))
                .setCreatedAt(Instant.ofEpochMilli(record.createdAt))
                .setUpdatedBy(Nullables.orEmpty(record.updatedBy))
                .setUpdatedAt(Instant.ofEpochMilli(record.updatedAt))
                .setVersion(record.version)
                .setRepeatNotifyDelay(Duration.ofMillis(record.repeatNotifyDelay))
                .setNotifyAboutStatus(decodeSubscribeOn(record.subscribeOn))
                .setLabels(decodeJsonMap(record.labels))
                .setDefaultForProject(record.defaultForProject)
                .setDefaultForSeverity(record.defaultForSeverity.stream().map(AlertSeverity::valueOf).collect(Collectors.toSet()))
                .build();
    }

    public Set<EvaluationStatus.Code> decodeSubscribeOn(int subscribeOn) {
        EnumSet<EvaluationStatus.Code> result = EnumSet.noneOf(EvaluationStatus.Code.class);
        for (EvaluationStatus.Code code : EvaluationStatus.Code.values()) {
            int mask = 1 << code.getNumber();
            if ((subscribeOn & mask) != 0) {
                result.add(code);
            }
        }
        return result;
    }

    private Notification.Builder<?, ?> decodeConfig(NotificationRecord record) {
        NotificationType type = NotificationType.forNumber(record.type);
        return decodeConfig(type, record.config);
    }

    public Notification.Builder<?, ?> decodeConfig(NotificationType type, String config) {
        try {
            return switch (type) {
                case EMAIL -> decodeEmailConfig(config);
                case WEBHOOK -> decodeWebhookConfig(config);
                case PHONE_CALL -> decodePhoneConfig(config);
                case JUGGLER -> decodeJugglerConfig(config);
                case SMS -> decodeSmsConfig(config);
                case TELEGRAM -> decodeTelegramConfig(config);
                case CLOUD_EMAIL -> decodeCloudEmailConfig(config);
                case CLOUD_SMS -> decodeCloudSmsConfig(config);
                case YA_CHATS -> decodeYaChatsConfig(config);
                case DATALENS_EMAIL -> decodeDatalensEmailConfig(config);
                case CLOUD_PUSH -> decodeCloudPushConfig(config);
                default -> throw new UnsupportedOperationException("Not supported notification type: " + type);
            };
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }

    public String encodeJsonMap(Map<String, String> map) {
        try {
            return mapper.writeValueAsString(map);
        } catch (JsonProcessingException e) {
            throw Throwables.propagate(e);
        }
    }

    public Map<String, String> decodeJsonMap(String str) {
        if (str.isEmpty()) {
            return Collections.emptyMap();
        }

        try {
            return mapper.readValue(str, new TypeReference<Map<String, String>>() {
            });
        } catch (IOException e) {
            throw Throwables.propagate(e);
        }
    }
}
