package ru.yandex.solomon.alert.dao.migrate;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;

import ru.yandex.solomon.alert.dao.EntitiesDao;
import ru.yandex.solomon.alert.dao.Entity;
import ru.yandex.solomon.alert.dao.SchemaAwareDao;
import ru.yandex.solomon.alert.dao.ydb.YdbExceptionHandler;
import ru.yandex.solomon.idempotency.IdempotentOperation;

/**
 * @author Vladimir Gordiychuk
 */
public class MigrateEntityDao<T extends Entity> implements EntitiesDao<T>, SchemaAwareDao {
    private final EntitiesDao<T> source;
    private final EntitiesDao<T> target;

    public MigrateEntityDao(EntitiesDao<T> source, EntitiesDao<T> target) {
        this.source = source;
        this.target = target;
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return target.createSchemaForTests();
    }

    @Override
    public CompletableFuture<?> createSchema(String projectId) {
        return target.createSchemaForTests()
            .thenCompose(ignore -> find(source, projectId))
            .handle((sourceMap, e) -> {
                if (e != null) {
                    if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                        // Previous version was already dropped
                        return CompletableFuture.completedFuture(null);
                    } else {
                        throw Throwables.propagate(e);
                    }
                }

                return find(target, projectId)
                    .thenCompose(targetMap -> migrateData(projectId, sourceMap, targetMap));
            })
            .thenCompose(future -> future);
    }

    private CompletableFuture<Map<String, T>> find(EntitiesDao<T> from, String projectId) {
        return from.findAll(projectId)
                .thenApply(list -> list.stream().collect(Collectors.toMap(Entity::getId, Function.identity())));
    }

    @Override
    public CompletableFuture<Optional<T>> insert(T entity, IdempotentOperation op) {
        return source.insert(entity, op)
                .exceptionally(this::ignoreNotExistsSchema)
                .thenCompose(ignore -> target.insert(entity, op));
    }

    @Override
    public CompletableFuture<Optional<T>> update(T entity, IdempotentOperation op) {
        return source.update(entity, op)
                .exceptionally(this::ignoreNotExistsSchema)
                .thenCompose(ignore -> target.update(entity, op));
    }

    @Override
    public CompletableFuture<Void> deleteById(String projectId, String id, IdempotentOperation op) {
        return source.deleteById(projectId, id, op)
                .exceptionally(this::ignoreNotExistsSchema)
                .thenCompose(ignore -> target.deleteById(projectId, id, op));
    }

    @Override
    public CompletableFuture<Void> deleteProject(String projectId) {
        return source.deleteProject(projectId)
            .thenCompose(ignore -> target.deleteProject(projectId));
    }

    @Override
    public CompletableFuture<Void> find(String projectId, Consumer<T> consumer) {
        return target.find(projectId, consumer);
    }

    @Override
    public CompletableFuture<Set<String>> findProjects() {
        return target.findProjects();
    }

    private CompletableFuture<?> migrateData(String projectId, Map<String, T> sourceMap, Map<String, T> targetMap) {
        // deletes
        List<String> deletes = new ArrayList<>();
        for (String id : targetMap.keySet()) {
            if (!sourceMap.containsKey(id)) {
                deletes.add(id);
            }
        }

        List<T> inserts = new ArrayList<>();
        List<T> updates = new ArrayList<>();
        for (T entity : sourceMap.values()) {
            T targetEntity = targetMap.get(entity.getId());
            if (targetEntity == null) {
                inserts.add(entity);
                continue;
            }

            if (targetEntity.getVersion() < entity.getVersion()) {
                updates.add(entity);
            }
        }

        return removeFromTarget(projectId, deletes)
                .thenCompose(ignore -> updateToTarget(updates))
                .thenCompose(ignore -> insertToTarget(inserts));
    }

    private CompletableFuture<?> removeFromTarget(String projectId, List<String> deletes) {
        return new MigrateSubProcess<>(deletes, id -> target.deleteById(projectId, id, IdempotentOperation.NO_OPERATION)).run();
    }

    private CompletableFuture<?> insertToTarget(List<T> inserts) {
        return new MigrateSubProcess<>(inserts, entity -> target.insert(entity, IdempotentOperation.NO_OPERATION)).run();
    }

    private CompletableFuture<?> updateToTarget(List<T> updates) {
        return new MigrateSubProcess<>(updates, entity -> target.update(entity, IdempotentOperation.NO_OPERATION)).run();
    }

    private <R> R ignoreNotExistsSchema(Throwable e) {
        if (YdbExceptionHandler.isPathDoesNotExist(e)) {
            return null;
        } else {
            throw Throwables.propagate(e);
        }
    }

    private static class MigrateSubProcess<T> {
        private final Function<T, CompletableFuture<?>> function;
        private final List<T> tasks;
        private final AtomicInteger index = new AtomicInteger();
        private final CompletableFuture<?> future;

        MigrateSubProcess(List<T> tasks, Function<T, CompletableFuture<?>> function) {
            this.function = function;
            this.tasks = tasks;
            this.future = new CompletableFuture<>();
        }

        CompletableFuture<?> run() {
            removeNext();
            return future;
        }

        private void removeNext() {
            if (index.get() >= tasks.size()) {
                future.complete(null);
                return;
            }

            function.apply(tasks.get(index.getAndIncrement()))
                    .whenComplete((ignore, e) -> {
                        if (e != null) {
                            future.completeExceptionally(e);
                            return;
                        }

                        removeNext();
                    });
        }
    }
}
