package ru.yandex.solomon.alert.dao.migrate;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;

import ru.yandex.solomon.alert.dao.SchemaAwareLocksDao;
import ru.yandex.solomon.alert.dao.ydb.YdbExceptionHandler;
import ru.yandex.solomon.locks.LockDetail;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MigrateLocksDao implements SchemaAwareLocksDao {
    private final SchemaAwareLocksDao source;
    private final SchemaAwareLocksDao target;

    public MigrateLocksDao(SchemaAwareLocksDao source, SchemaAwareLocksDao target) {
        this.source = source;
        this.target = target;
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return target.createSchemaForTests();
    }

    @Override
    public CompletableFuture<LockDetail> acquireLock(String lockId, String owner, Instant expiredAt) {
        return source.acquireLock(lockId, owner, expiredAt)
                .handle((sourceDetails, e) -> {
                    if (e != null) {
                        if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                            // table already dropped
                            return target.acquireLock(lockId, owner, expiredAt);
                        } else {
                            throw Throwables.propagate(e);
                        }
                    }

                    // acquire failed no need acquire into target version
                    if (!sourceDetails.owner().equals(owner)) {
                        return CompletableFuture.completedFuture(sourceDetails);
                    }

                    return target.acquireLock(lockId, owner, expiredAt)
                            .thenApply(targetDetails -> ensureLockedBoth(owner, sourceDetails, targetDetails));
                })
                .thenCompose(future -> future);
    }

    private LockDetail ensureLockedBoth(String expectedOwner, LockDetail source, LockDetail target) {
        if (!expectedOwner.equals(source.owner())) {
            // prefer minimal version because not whole cluster updated yet
            return source;
        }

        if (!source.owner().equals(target.owner())) {
            throw new IllegalStateException("Lock acquire only into one table: source - " + source + "; target - " + target);
        }

        return source;
    }

    @Override
    public CompletableFuture<Optional<LockDetail>> readLock(String lockId) {
        return source.readLock(lockId)
                .handle((result, e) -> {
                    if (e != null) {
                        if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                            return target.readLock(lockId);
                        } else {
                            throw Throwables.propagate(e);
                        }
                    }

                    return CompletableFuture.completedFuture(result);
                })
                .thenCompose(future -> future);
    }

    @Override
    public CompletableFuture<Boolean> extendLockTime(String lockId, String owner, Instant expiredAt) {
        return source.extendLockTime(lockId, owner, expiredAt)
                .handle((sourceResult, e) -> {
                    if (e != null) {
                        if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                            // table already dropped
                            return target.extendLockTime(lockId, owner, expiredAt);
                        } else {
                            throw Throwables.propagate(e);
                        }
                    }

                    return target.extendLockTime(lockId, owner, expiredAt)
                            .thenCompose(targetResult -> {
                                if (sourceResult && targetResult) {
                                    return CompletableFuture.completedFuture(Boolean.TRUE);
                                } else {
                                    return releaseLock(lockId, owner)
                                            .thenApply(ignore -> Boolean.FALSE);
                                }
                            });
                })
                .thenCompose(future -> future);
    }

    @Override
    public CompletableFuture<Boolean> releaseLock(String lockId, String owner) {
        return source.releaseLock(lockId, owner)
                .handle((sourceResult, e) -> {
                    if (e != null) {
                        if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                            // table already dropped
                            return target.releaseLock(lockId, owner);
                        } else {
                            throw Throwables.propagate(e);
                        }
                    }

                    return target.releaseLock(lockId, owner)
                            .thenApply(targetResult -> sourceResult || targetResult);
                })
                .thenCompose(future -> future);
    }

    @Override
    public CompletableFuture<List<LockDetail>> listLocks() {
        return source.listLocks()
                .handle((sourceResult, e) -> {
                    if (e != null) {
                        if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                            // table already dropped
                            return target.listLocks();
                        } else {
                            throw Throwables.propagate(e);
                        }
                    }

                    return target.listLocks()
                            .thenApply(targetResult -> {
                                return List.copyOf(Stream.concat(sourceResult.stream(), targetResult.stream())
                                        .collect(Collectors.toMap(LockDetail::id, Function.identity(),(left, right) -> left))
                                        .values());
                            });
                })
                .thenCompose(future -> future);
    }
}
