package ru.yandex.solomon.alert.dao.migrate;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import com.google.common.base.Throwables;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;

import ru.yandex.solomon.alert.dao.TelegramDao;
import ru.yandex.solomon.alert.dao.TelegramRecord;
import ru.yandex.solomon.alert.dao.ydb.YdbExceptionHandler;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class MigrateTelegramDao implements TelegramDao {
    private final TelegramDao source;
    private final TelegramDao target;

    public MigrateTelegramDao(TelegramDao source, TelegramDao target) {
        this.source = source;
        this.target = target;
    }

    @Override
    public CompletableFuture<Void> migrate() {
        return source.findAll()
            .handle((sourceList, e) -> {
                if (e != null) {
                    if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                        // Previous version was already dropped
                        return completedFuture(null);
                    } else {
                        throw Throwables.propagate(e);
                    }
                }

                return target.findAll()
                    .thenCompose(targetList -> migrateData(sourceList, targetList));
            })
            .thenCompose(f -> f)
            .thenAccept(ignore -> {});
    }

    @Override
    public CompletableFuture<Void> upsert(TelegramRecord record) {
        return source.upsert(record)
            .exceptionally(this::ignoreNotExistsSchema)
            .thenCompose(ignore -> target.upsert(record));
    }

    @Override
    public CompletableFuture<Optional<TelegramRecord>> get(long chatId) {
        return source.get(chatId)
            .handle((r, e) -> {
                if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                    // Previous version was already dropped
                    return target.get(chatId);
                } else if (e != null) {
                    throw Throwables.propagate(e);
                }

                return completedFuture(r);
            })
            .thenCompose(f -> f);
    }

    @Override
    public CompletableFuture<Void> deleteById(long chatId) {
        return source.deleteById(chatId)
            .exceptionally(this::ignoreNotExistsSchema)
            .thenCompose(ignore -> target.deleteById(chatId));
    }

    @Override
    public CompletableFuture<List<TelegramRecord>> findAll() {
        return source.findAll()
            .handle((list, e) -> {
                if (YdbExceptionHandler.isPathDoesNotExist(e)) {
                    // Previous version was already dropped
                    return target.findAll();
                } else if (e != null) {
                    throw Throwables.propagate(e);
                }

                return completedFuture(list);
            })
            .thenCompose(f -> f);
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return target.createSchemaForTests();
    }

    private Void ignoreNotExistsSchema(Throwable e) {
        if (YdbExceptionHandler.isPathDoesNotExist(e)) {
            return null;
        } else {
            throw Throwables.propagate(e);
        }
    }

    private CompletableFuture<Void> migrateData(List<TelegramRecord> sourceList, List<TelegramRecord> targetList) {
        // deletes???
        LongOpenHashSet sourceIds = new LongOpenHashSet(sourceList.size());
        for (var record : sourceList) {
            sourceIds.add(record.getChatId());
        }

        // deletes
        LongArrayList deletes = new LongArrayList();
        for (var record : targetList) {
            if (!sourceIds.contains(record.getChatId())) {
                deletes.add(record.getChatId());
            }
        }

        return removeFromTarget(deletes)
            .thenCompose(ignore -> upsertToTarget(sourceList));
    }

    private CompletableFuture<Void> removeFromTarget(LongArrayList deletes) {
        return new MigrateSubProcess<>(deletes, target::deleteById).run();
    }

    private CompletableFuture<Void> upsertToTarget(List<TelegramRecord> inserts) {
        return new MigrateSubProcess<>(inserts, target::upsert).run();
    }

    private static class MigrateSubProcess<T> {
        private final Function<T, CompletableFuture<?>> function;
        private final List<T> tasks;
        private final AtomicInteger index = new AtomicInteger();
        private final CompletableFuture<Void> future;

        MigrateSubProcess(List<T> tasks, Function<T, CompletableFuture<?>> function) {
            this.function = function;
            this.tasks = tasks;
            this.future = new CompletableFuture<>();
        }

        CompletableFuture<Void> run() {
            removeNext();
            return future;
        }

        private void removeNext() {
            if (index.get() >= tasks.size()) {
                future.complete(null);
                return;
            }

            function.apply(tasks.get(index.getAndIncrement()))
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        future.completeExceptionally(e);
                        return;
                    }

                    removeNext();
                });
        }
    }
}
