package ru.yandex.solomon.alert.dao.ydb;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Value;

import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.ydb.queries.AlertTemplateQueries;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.domain.NoPointsPolicy;
import ru.yandex.solomon.alert.domain.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.domain.threshold.Compare;
import ru.yandex.solomon.alert.domain.threshold.TargetStatus;
import ru.yandex.solomon.alert.domain.threshold.ThresholdType;
import ru.yandex.solomon.alert.template.domain.AbstractAlertTemplateBuilder;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateId;
import ru.yandex.solomon.alert.template.domain.AlertTemplateLastVersion;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter;
import ru.yandex.solomon.alert.template.domain.AlertTemplateType;
import ru.yandex.solomon.alert.template.domain.expression.ExpressionAlertTemplate;
import ru.yandex.solomon.alert.template.domain.threshold.TemplatePredicateRule;
import ru.yandex.solomon.alert.template.domain.threshold.ThresholdAlertTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.json;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toJson;
import static ru.yandex.solomon.core.db.dao.kikimr.QueryText.paging;

/**
 * @author Alexey Trushkin
 */
public class YdbAlertTemplateDao implements AlertTemplateDao {

    public static final Comparator<AlertTemplate> TEMPLATE_COMPARATOR = Comparator.comparing(AlertTemplate::getServiceProviderId)
            .thenComparing(AlertTemplate::getName);
    private static final StructType TYPE = StructType.of(
            "id", PrimitiveType.utf8(),
            "templateVersionTag", PrimitiveType.utf8());
    private static final ListType LIST_TYPE = ListType.of(TYPE);
    private static final String ALERT_TEMPLATES_TABLE = "AlertTemplates";
    private final String root;
    private final String tablePath;
    private final SchemeClient schemeClient;
    private final Table table;
    private final String findQuery;
    private final String findVersionsPagedQuery;
    private final String findTemplatesPagedQuery;
    private final String insertQuery;
    private final String insertPublishedQuery;
    private final String insertPublishedWithUpdateQuery;
    private final String findListQuery;

    public YdbAlertTemplateDao(
            String root,
            TableClient tableClient,
            SchemeClient schemeClient,
            YdbSchemaVersion version,
            ObjectMapper objectMapper)
    {
        this.root = root + "/Alerting/" + version.folderName();
        this.schemeClient = schemeClient;
        this.tablePath = this.root + "/" + ALERT_TEMPLATES_TABLE;
        this.table = new Table(tableClient, tablePath, objectMapper);

        this.insertQuery = String.format(AlertTemplateQueries.INSERT_TEMPLATE, tablePath);
        this.insertPublishedQuery = String.format(AlertTemplateQueries.INSERT_PUBLISHED_TEMPLATE,
                this.root,
                this.root);
        this.insertPublishedWithUpdateQuery = String.format(AlertTemplateQueries.INSERT_PUBLISHED_WITH_UPDATE_TEMPLATE,
                this.root, this.root, this.root);
        this.findQuery = String.format(AlertTemplateQueries.FIND_TEMPLATE, tablePath);
        this.findListQuery = String.format(AlertTemplateQueries.FIND_LIST_TEMPLATE, LIST_TYPE, tablePath);
        this.findVersionsPagedQuery = String.format(AlertTemplateQueries.FIND_TEMPLATE_VERSIONS_TEMPLATE, tablePath);
        this.findTemplatesPagedQuery = String.format(AlertTemplateQueries.FIND_TEMPLATES_TEMPLATE, this.root, this.root);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> schemeClient.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<Boolean> create(AlertTemplate alertTemplate) {
        try {
            return table.insertOne(insertQuery, alertTemplate);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<AlertTemplate>> findById(String id, String templateVersionTag) {
        try {
            Params params = Params.of(
                    "$id", utf8(id),
                    "$templateVersionTag", utf8(templateVersionTag)
            );
            return table.queryOne(findQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<AlertTemplate>> getAll() {
        try {
            return table.queryAll();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<AlertTemplate>> findVersions(List<AlertTemplateLastVersion> items) {
        try {
            if (items.isEmpty()) {
                return CompletableFuture.completedFuture(List.of());
            }
            List<Value> values = new ArrayList<>(items.size());
            for (var item : items) {
                values.add(TYPE.newValue(
                        "id", utf8(item.id()),
                        "templateVersionTag", utf8(item.templateVersionTag())));
            }
            Params params = Params.of("$rows", LIST_TYPE.newValue(values));
            return table.queryList(findListQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> publish(AlertTemplate config, int version) {
        try {
            Params params = table.toParams(config);
            params.put("$publishingTaskId", utf8(""));
            if (version == -1) {
                return table.queryVoid(insertPublishedQuery, params)
                        .thenApply(unused -> true);
            }
            params.put("$version", uint32(version));
            return table.queryBool(insertPublishedWithUpdateQuery, params)
                    .thenCompose(updated -> updated ?
                            CompletableFuture.completedFuture(true) :
                            CompletableFuture.failedFuture(new ConflictException("AlertTemplate publication failed with conflict")));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<AlertTemplate>> listTemplateVersions(String templateId, int pageSize, String pageToken) {
        try {
            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$id", utf8(templateId),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(findVersionsPagedQuery, params)
                    .thenApply(result -> {
                        final String nextPageToken;
                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }
                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<TokenBasePage<AlertTemplate>> listLastTemplates(
            String serviceProviderId,
            String nameFilter,
            String labelsSelector,
            int pageSize,
            String pageToken,
            Predicate<AlertTemplate> skip)
    {
        try {
            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);
            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }
            Params params = Params.of(
                    "$serviceProviderId", utf8(serviceProviderId),
                    "$name", utf8(nameFilter)
            );
            var selectors = labelsSelector.isEmpty() ?  Selectors.of() : Selectors.parse(labelsSelector);
            return table.queryPage(params, PageOptions.ALL, (opts) -> {
                return QueryText.prepareQuery(findTemplatesPagedQuery, paging(opts));
            }).thenApply(alertTemplatePagedResult -> {
                Set<String> values = new HashSet<>(alertTemplatePagedResult.getResult().size());
                var templates = alertTemplatePagedResult.getResult().stream()
                        .filter(alertTemplate -> labelsSelector.isEmpty() || selectors.match(alertTemplate.getLabels()))
                        .filter(alertTemplate -> !skip.test(alertTemplate) && values.add(alertTemplate.getId()))
                        .sorted(TEMPLATE_COMPARATOR)
                        .skip(offset)
                        .collect(Collectors.toList());

                final String nextPageToken;
                if (templates.size() > size) {
                    nextPageToken = String.valueOf(offset + size);
                } else {
                    nextPageToken = "";
                }
                return new TokenBasePage<>(templates.stream().limit(size).collect(Collectors.toList()), nextPageToken);
            });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private static final class Table extends YdbTable<AlertTemplateId, AlertTemplate> {

        private final ObjectMapper objectMapper;

        Table(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.objectMapper = objectMapper;
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .addNullableColumn("templateVersionTag", PrimitiveType.utf8())
                    .addNullableColumn("alertTemplateType", PrimitiveType.utf8())
                    .addNullableColumn("serviceProviderId", PrimitiveType.utf8())
                    .addNullableColumn("name", PrimitiveType.utf8())
                    .addNullableColumn("description", PrimitiveType.utf8())
                    .addNullableColumn("data", PrimitiveType.json())
                    .addNullableColumn("createdAt", PrimitiveType.int64())
                    .addNullableColumn("updatedAt", PrimitiveType.int64())
                    .addNullableColumn("createdBy", PrimitiveType.utf8())
                    .addNullableColumn("updatedBy", PrimitiveType.utf8())
                    .setPrimaryKeys("id", "templateVersionTag")
                    .build();
        }

        @Override
        protected AlertTemplateId getId(AlertTemplate config) {
            return config.getCompositeId();
        }

        @Override
        protected Params toParams(AlertTemplate config) {
            var data = toJson(objectMapper, config);

            return Params.create()
                    .put("$id", utf8(config.getId()))
                    .put("$templateVersionTag", utf8(config.getTemplateVersionTag()))
                    .put("$alertTemplateType", utf8(config.getAlertTemplateType().name()))
                    .put("$serviceProviderId", utf8(config.getServiceProviderId()))
                    .put("$name", utf8(config.getName()))
                    .put("$description", utf8(config.getDescription()))
                    .put("$data", json(data))
                    .put("$createdAt", int64(config.getCreatedAt().toEpochMilli()))
                    .put("$updatedAt", int64(config.getUpdatedAt().toEpochMilli()))
                    .put("$createdBy", utf8(config.getCreatedBy()))
                    .put("$updatedBy", utf8(config.getUpdatedBy()));
        }

        @Override
        protected AlertTemplate mapFull(ResultSetReader r) {
            try {
                return toTemplate(r);
            } catch (IOException ioException) {
                throw new UncheckedIOException(ioException);
            }
        }

        @Override
        protected AlertTemplate mapPartial(ResultSetReader r) {
            try {
                return toTemplate(r);
            } catch (IOException ioException) {
                throw new UncheckedIOException(ioException);
            }
        }

        private AlertTemplate toTemplate(ResultSetReader r) throws IOException {
            String data = r.getColumn("data").getJson();
            JsonNode root = objectMapper.readTree(data);

            var alertBuilder = decodeConfig(r, root)
                    .setId(r.getColumn("id").getUtf8())
                    .setTemplateVersionTag(r.getColumn("templateVersionTag").getUtf8())
                    .setServiceProviderId(r.getColumn("serviceProviderId").getUtf8())
                    .setCreatedBy(r.getColumn("createdBy").getUtf8())
                    .setUpdatedBy(r.getColumn("updatedBy").getUtf8())
                    .setCreatedAt(Instant.ofEpochMilli(r.getColumn("createdAt").getInt64()))
                    .setUpdatedAt(Instant.ofEpochMilli(r.getColumn("updatedAt").getInt64()))
                    .setPeriodMillis(root.get("periodMillis").asInt())
                    .setName(r.getColumn("name").getUtf8())
                    .setDescription(r.getColumn("description").getUtf8())
                    .setGroupByLabels(decodeJsonArray(root.get("groupByLabels")))
                    .setParameters(decodeJsonParamsArray(root.get("parameters")))
                    .setThresholds(decodeJsonParamsArray(root.get("thresholds")))
                    .setGroupByLabels(decodeJsonArray(root.get("groupByLabels")))
                    .setAnnotations(decodeJsonMap(root.get("annotations")))
                    .setLabels(decodeJsonMap(root.get("labels")))
                    .setDelaySeconds(root.get("delaySeconds").asInt())
                    .setDefaultTemplate(root.get("defaultTemplate") != null
                            ? root.get("defaultTemplate").asBoolean()
                            : false);

            String resolvedEmptyPolicy = root.get("resolvedEmptyPolicy").asText();
            if (!Strings.isNullOrEmpty(resolvedEmptyPolicy)) {
                alertBuilder.setResolvedEmptyPolicy(ResolvedEmptyPolicy.valueOf(resolvedEmptyPolicy));
            }
            String noPointsPolicy = root.get("noPointsPolicy").asText();
            if (!Strings.isNullOrEmpty(noPointsPolicy)) {
                alertBuilder.setNoPointsPolicy(NoPointsPolicy.valueOf(noPointsPolicy));
            }

            String severity = root.get("severity") == null
                    ? "UNKNOWN"
                    : root.get("severity").asText();
            alertBuilder.setAlertSeverity(AlertSeverity.valueOf(severity));

            return alertBuilder.build();
        }

        private AbstractAlertTemplateBuilder<?, ?> decodeConfig(ResultSetReader r, JsonNode root) {
            AlertTemplateType alertTemplateType = AlertTemplateType.valueOf(r.getColumn("alertTemplateType").getUtf8());
            return decodeConfig(alertTemplateType, root);
        }

        public AbstractAlertTemplateBuilder<?, ?> decodeConfig(AlertTemplateType type, JsonNode root) {
            try {
                switch (type) {
                    case EXPRESSION:
                        return decodeExpressionConfig(root);
                    case THRESHOLD:
                        return decodeThresholdConfig(root);
                    default:
                        throw new UnsupportedOperationException("Not supported alert template type: " + type);
                }
            } catch (IOException e) {
                throw Throwables.propagate(e);
            }
        }

        private ExpressionAlertTemplate.Builder decodeExpressionConfig(JsonNode root) throws IOException {
            return ExpressionAlertTemplate.newBuilder()
                    .setProgram(root.get("program").asText());
        }

        private TemplatePredicateRule decodePredicateRule(JsonNode item) {
            return TemplatePredicateRule.onThreshold(item.get("threshold").asDouble())
                    .withThresholdType(ThresholdType.valueOf(item.get("thresholdType").asText()))
                    .withComparison(Compare.valueOf(item.get("comparison").asText()))
                    .withTargetStatus(TargetStatus.valueOf(item.get("targetStatus").asText()))
                    .onThresholdParameter(item.get("thresholdParameterTemplate") == null ? "" : item.get("thresholdParameterTemplate").asText());
        }

        private ThresholdAlertTemplate.Builder decodeThresholdConfig(JsonNode root) throws IOException {
            ThresholdAlertTemplate.Builder builder = ThresholdAlertTemplate.newBuilder()
                    .setSelectors(root.get("selectors").asText());

            if (root.has("transformations")) {
                builder.setTransformations(root.get("transformations").asText());
            }

            if (root.has("predicateRules")) {
                JsonNode predicateRules = root.get("predicateRules");
                Preconditions.checkState(predicateRules.isArray());
                List<TemplatePredicateRule> rules = new ArrayList<>(predicateRules.size());
                for (JsonNode item : predicateRules) {
                    rules.add(decodePredicateRule(item));
                }
                builder.setPredicateRules(rules.stream());
            }

            return builder;
        }

        public List<String> decodeJsonArray(JsonNode root) {
            Preconditions.checkState(root.isArray());
            List<String> result = new ArrayList<>(root.size());
            for (JsonNode item : root) {
                result.add(item.asText());
            }

            return result;
        }

        public List<AlertTemplateParameter> decodeJsonParamsArray(JsonNode root) {
            Preconditions.checkState(root.isArray());
            List<AlertTemplateParameter> result = new ArrayList<>(root.size());
            for (JsonNode item : root) {
                result.add(objectMapper.convertValue(item, AlertTemplateParameter.class));
            }

            return result;
        }

        public Map<String, String> decodeJsonMap(JsonNode node) {
            if (node == null) {
                return Collections.emptyMap();
            }
            return objectMapper.convertValue(node, new TypeReference<Map<String, String>>() {
            });
        }
    }
}
