package ru.yandex.solomon.alert.dao.ydb;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.ydb.queries.AlertTemplateLastVersionQueries;
import ru.yandex.solomon.alert.template.domain.AlertTemplateLastVersion;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.TokenBasePage;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Alexey Trushkin
 */
public class YdbAlertTemplateLastVersionDao implements AlertTemplateLastVersionDao {

    protected static final String TABLE_NAME = "AlertTemplateLastVersions";
    private final String root;
    private final String tablePath;
    private final SchemeClient schemeClient;
    private final Table table;
    private final String findQuery;
    private final String findOneQuery;
    private final String insertQuery;
    private final String updateTaskQuery;
    private final String deleteQuery;

    public YdbAlertTemplateLastVersionDao(
            String root,
            TableClient tableClient,
            SchemeClient schemeClient,
            YdbSchemaVersion version)
    {
        this.root = root + "/Alerting/" + version.folderName();
        this.schemeClient = schemeClient;
        this.tablePath = this.root + "/" + TABLE_NAME;
        this.table = new Table(tableClient, tablePath);

        this.findOneQuery = String.format(AlertTemplateLastVersionQueries.FIND_ONE_TEMPLATE, tablePath);
        this.findQuery = String.format(AlertTemplateLastVersionQueries.FIND_VERSIONS_TEMPLATE, tablePath);
        this.insertQuery = String.format(AlertTemplateLastVersionQueries.INSERT_TEMPLATE, tablePath);
        this.updateTaskQuery = String.format(AlertTemplateLastVersionQueries.UPDATE_TASK_TEMPLATE, tablePath, tablePath);
        this.deleteQuery = String.format(AlertTemplateLastVersionQueries.DELETE_TEMPLATE, tablePath, tablePath);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> schemeClient.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<TokenBasePage<AlertTemplateLastVersion>> find(String serviceProviderId, String name, int pageSize, String pageToken) {
        try {
            final int size;
            if (pageSize <= 0) {
                size = 100;
            } else {
                size = Math.min(pageSize, 1000);
            }

            int offset = pageToken.isEmpty() ? 0 : Integer.parseInt(pageToken);

            Params params = Params.of(
                    "$serviceProviderId", utf8(serviceProviderId),
                    "$name", utf8(name),
                    "$pageSize", int32(size + 1),
                    "$pageOffset", int32(offset));

            return table.queryList(findQuery, params)
                    .thenApply(result -> {
                        final String nextPageToken;
                        if (result.size() > size) {
                            result = result.subList(0, size);
                            nextPageToken = String.valueOf(offset + size);
                        } else {
                            nextPageToken = "";
                        }
                        return new TokenBasePage<>(result, nextPageToken);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<AlertTemplateLastVersion>> findById(String id) {
        try {
            Params params = Params.of(
                    "$id", utf8(id)
            );
            return table.queryOne(findOneQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> delete(String id, int version) {
        try {
            Params params = Params.of(
                    "$id", utf8(id),
                    "$version", uint32(version)
            );
            return table.queryBool(deleteQuery, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> create(AlertTemplateLastVersion alertTemplateLastVersion) {
        try {
            return table.insertOne(insertQuery, alertTemplateLastVersion);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Boolean> updateDeployTask(String templateId, int version, String taskId) {
        try {
            Params params = Params.create()
                    .put("$publishingTaskId", utf8(taskId))
                    .put("$version", uint32(version))
                    .put("$id", utf8(templateId));
            return table.queryBool(updateTaskQuery, params)
                    .thenCompose(updated -> updated ?
                            CompletableFuture.completedFuture(true) :
                            CompletableFuture.failedFuture(new ConflictException("AlertTemplateLastVersion update failed with conflict")));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<AlertTemplateLastVersion>> getAll() {
        try {
            return table.queryAll();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private static final class Table extends YdbTable<String, AlertTemplateLastVersion> {

        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("id", PrimitiveType.utf8())
                    .addNullableColumn("templateVersionTag", PrimitiveType.utf8())
                    .addNullableColumn("serviceProviderId", PrimitiveType.utf8())
                    .addNullableColumn("name", PrimitiveType.utf8())
                    .addNullableColumn("version", PrimitiveType.uint32())
                    .addNullableColumn("publishingTaskId", PrimitiveType.utf8())
                    .setPrimaryKeys("id")
                    .build();
        }

        @Override
        protected String getId(AlertTemplateLastVersion config) {
            return config.id();
        }

        @Override
        protected Params toParams(AlertTemplateLastVersion config) {
            return Params.create()
                    .put("$id", utf8(config.id()))
                    .put("$templateVersionTag", utf8(config.templateVersionTag()))
                    .put("$serviceProviderId", utf8(config.serviceProviderId()))
                    .put("$name", utf8(config.name()))
                    .put("$version", uint32(config.version()))
                    .put("$publishingTaskId", utf8(config.publishingTaskId()));
        }

        @Override
        protected AlertTemplateLastVersion mapFull(ResultSetReader r) {
            return new AlertTemplateLastVersion(
                    r.getColumn("id").getUtf8(),
                    r.getColumn("templateVersionTag").getUtf8(),
                    r.getColumn("serviceProviderId").getUtf8(),
                    r.getColumn("name").getUtf8(),
                    (int) r.getColumn("version").getUint32(),
                    r.getColumn("publishingTaskId").getUtf8());
        }

        @Override
        protected AlertTemplateLastVersion mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}

