package ru.yandex.solomon.alert.dao.ydb.entity;

import java.nio.file.Path;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.alert.dao.AlertsDao;
import ru.yandex.solomon.alert.dao.codec.AlertCodec;
import ru.yandex.solomon.alert.dao.codec.AlertRecord;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.inject.spring.AlertingIdempotentOperationContext;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.idempotency.IdempotentOperationExistException;
import ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationDao;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbAlertsDao implements AlertsDao {
    private static final int SELECT_LIMIT = 1000;
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
        YdbAlertsDao.class,
        "alerts",
        List.of(
            "delete_all",
            "delete_one",
            "insert",
            "update",
            "select_all_limited",
            "operation"
        ));

    private final String tablePath;
    private final AlertsTable table;
    private final SchemeClient scheme;
    private final QueryText queryText;

    public YdbAlertsDao(String path, TableClient tableClient, SchemeClient schemeClient, YdbSchemaVersion version, ObjectMapper objectMapper) {
        this.tablePath = path + "/Alerting/" + version.folderName() + "/Alerts";
        this.table = new AlertsTable(tableClient, tablePath, objectMapper);
        this.scheme = schemeClient;
        var operationsPath = path + AlertingIdempotentOperationContext.PATH + "/IdempotentOperation";
        this.queryText = TEMPLATE.build(Map.of("alerts.table.path", tablePath, "limit", SELECT_LIMIT, "idempotentOperation.table.path", operationsPath));
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return scheme.makeDirectories(Path.of(tablePath).getParent().toString())
            .thenAccept(status -> status.expect("parent directories success created"))
            .thenCompose(ignore -> scheme.describePath(tablePath))
            .thenCompose(exist -> !exist.isSuccess()
                ? table.create()
                : completedFuture(null));
    }

    @Override
    public CompletableFuture<?> createSchema(String projectId) {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Optional<Alert>> insert(Alert entity, IdempotentOperation op) {
        Params params = table.toParams(entity);
        YdbIdempotentOperationDao.fillExternalParams(params, op);
        try {
            return table.execute(queryText.query("insert"), params)
                    .thenCompose(result -> handleQueryResult(op, result));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<Alert>> update(Alert entity, IdempotentOperation op) {
        Params params = table.toParams(entity);
        YdbIdempotentOperationDao.fillExternalParams(params, op);
        try {
            return table.execute(queryText.query("update"), params)
                    .thenCompose(result -> handleQueryResult(op, result));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Optional<Alert>> handleQueryResult(IdempotentOperation op, Result<DataQueryResult> result) {
        if (!result.isSuccess() && result.getCode() == StatusCode.PRECONDITION_FAILED && !op.isNoOperation()) {
            return ensureOperationDone(op, result);
        }
        // read result
        var data = result.expect("alert creation failed");
        if (data.getResultSetCount() == 0) {
            return CompletableFuture.completedFuture(Optional.empty());
        }
        ResultSetReader resultSet = data.getResultSet(0);
        if (!resultSet.next()) {
            return CompletableFuture.completedFuture(Optional.empty());
        }
        return CompletableFuture.completedFuture(Optional.of(table.mapFull(resultSet)));
    }

    @Override
    public CompletableFuture<Void> deleteById(String projectId, String id, IdempotentOperation op) {
        try {
            Params params = Params.create()
                    .put("$id", utf8(id))
                    .put("$projectId", utf8(projectId));
            YdbIdempotentOperationDao.fillExternalParams(params, op);
            return table.execute(queryText.query("delete_one"), params)
                    .thenCompose(result -> handleQueryResult(op, result))
                    .thenApply(optionalAlert -> null);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteProject(String projectId) {
        try {
            String query = queryText.query("delete_all");
            Params params = Params.of("$projectId", utf8(projectId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> find(String projectId, Consumer<Alert> consumer) {
        return findAllBySelect(projectId, consumer)
                .thenCompose(truncated -> {
                    if (!truncated) {
                        return completedFuture(null);
                    }

                    return findAllByReadTable(projectId, consumer);
                });
    }

    @Override
    public CompletableFuture<Set<String>> findProjects() {
        var settings = ReadTableSettings.newBuilder()
                .timeout(1, TimeUnit.MINUTES)
                .column("projectId")
                .build();

        Set<String> result = new HashSet<>();
        Predicate<ResultSetReader> predicate = rs -> {
            result.add(rs.getColumn(0).getUtf8());
            return false;
        };
        return table.queryAll(predicate, settings).thenApply(ignore -> result);
    }

    private CompletableFuture<Boolean> findAllBySelect(String projectId, Consumer<Alert> consumer) {
        try {
            String query = queryText.query("select_all_limited");
            Params params = Params.of("$projectId", utf8(projectId));
            return table.executeAndExpectSuccess(query, params)
                    .thenApply(result -> {
                        var resultSet = result.getResultSet(0);
                        if (resultSet.getRowCount() == SELECT_LIMIT || resultSet.isTruncated()) {
                            return Boolean.TRUE;
                        }

                        while (resultSet.next()) {
                            consumer.accept(table.mapFull(resultSet));
                        }

                        return Boolean.FALSE;
                    });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> findAllByReadTable(String projectId, Consumer<Alert> consumer) {
        var settings = ReadTableSettings.newBuilder()
                .timeout(1, TimeUnit.MINUTES)
                .fromKeyInclusive(utf8(projectId))
                .toKeyInclusive(utf8(projectId))
                .build();

        return table.queryAll(ignore -> true, settings)
                .thenAccept(alerts -> alerts.forEach(consumer));
    }

    private <T> CompletableFuture<T> ensureOperationDone(IdempotentOperation op, Result<DataQueryResult> parentResult) {
        return table.queryBool(queryText.query("operation"), YdbIdempotentOperationDao.existParams(op))
                .thenApply(result -> {
                    if (result) {
                        // handle if needed
                        throw new IdempotentOperationExistException();
                    }
                    // here is exception
                    parentResult.expect("alert creation failed, no idempotent operation");
                    return null;
                });
    }

    /**
     * ALERTS TABLE
     */
    private static final class AlertsTable extends YdbTable<AlertKey, Alert> {
        private final AlertCodec codec;

        AlertsTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.codec = new AlertCodec(objectMapper);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("type", PrimitiveType.uint32())
                .addNullableColumn("delaySeconds", PrimitiveType.uint32())
                .addNullableColumn("state", PrimitiveType.uint32())
                .addNullableColumn("annotations", PrimitiveType.utf8())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("groupByLabels", PrimitiveType.utf8())
                .addNullableColumn("notificationChannels", PrimitiveType.utf8())
                .addNullableColumn("config", PrimitiveType.utf8())
                .addNullableColumn("notificationConfig", PrimitiveType.utf8())
                .addNullableColumn("resolvedEmptyPolicy", PrimitiveType.utf8())
                .addNullableColumn("noPointsPolicy", PrimitiveType.utf8())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("version", PrimitiveType.uint32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected AlertKey getId(Alert alert) {
            return alert.getKey();
        }

        @Override
        protected Params toParams(Alert alert) {
            var record = codec.encode(alert);
            return Params.create()
                .put("$projectId", utf8(record.projectId))
                .put("$id", utf8(record.id))
                .put("$folderId", utf8(record.folderId))
                .put("$name", utf8(record.name))
                .put("$description", utf8(record.description))
                .put("$type", uint32(record.type))
                .put("$delaySeconds", uint32(record.delaySeconds))
                .put("$state", uint32(record.state))
                .put("$annotations", utf8(record.annotations))
                .put("$labels", utf8(record.labels))
                .put("$groupByLabels", utf8(record.groupByLabels))
                .put("$notificationChannels", utf8(record.notificationChannels))
                .put("$config", utf8(record.config))
                .put("$notificationConfig", utf8(record.notificationConfig))
                .put("$resolvedEmptyPolicy", utf8(record.resolvedEmptyPolicy))
                .put("$noPointsPolicy", utf8(record.noPointsPolicy))
                .put("$createdBy", utf8(record.createdBy))
                .put("$updatedBy", utf8(record.updatedBy))
                .put("$createdAt", int64(record.createdAt))
                .put("$updatedAt", int64(record.updatedAt))
                .put("$version", uint32(record.version));
        }

        @Override
        protected Alert mapFull(ResultSetReader r) {
            var record = new AlertRecord();
            record.projectId = r.getColumn("projectId").getUtf8();
            record.id = r.getColumn("id").getUtf8();
            record.folderId = r.getColumn("folderId").getUtf8();
            record.name = r.getColumn("name").getUtf8();
            record.description = r.getColumn("description").getUtf8();
            record.type = (int) r.getColumn("type").getUint32();
            record.delaySeconds = (int) r.getColumn("delaySeconds").getUint32();
            record.state = (int) r.getColumn("state").getUint32();
            record.annotations = r.getColumn("annotations").getUtf8();
            record.labels = r.getColumn("labels").getUtf8();
            record.groupByLabels = r.getColumn("groupByLabels").getUtf8();
            record.notificationChannels = r.getColumn("notificationChannels").getUtf8();
            record.config = r.getColumn("config").getUtf8();
            record.notificationConfig = r.getColumn("notificationConfig").getUtf8();
            record.resolvedEmptyPolicy = r.getColumn("resolvedEmptyPolicy").getUtf8();
            record.noPointsPolicy = r.getColumn("noPointsPolicy").getUtf8();
            record.createdBy = r.getColumn("createdBy").getUtf8();
            record.updatedBy = r.getColumn("updatedBy").getUtf8();
            record.createdAt = r.getColumn("createdAt").getInt64();
            record.updatedAt = r.getColumn("updatedAt").getInt64();
            record.version = (int) r.getColumn("version").getUint32();
            return codec.decode(record);
        }

        @Override
        protected Alert mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }

    public CompletableFuture<List<Alert>> listAll() {
        return table.queryAll();
    }
}
