package ru.yandex.solomon.alert.dao.ydb.entity;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;
import io.grpc.Status;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.dao.StatesCompressor;
import ru.yandex.solomon.alert.protobuf.TPersistAlertState;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbAlertsStateFilesDao implements AlertStatesFilesDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
        YdbTelegramDao.class,
        "alert_state_files",
        List.of(
            "delete_all",
            "delete_file",
            "find",
            "insert"
        ));

    private final String tablePath;
    private final FileTable table;
    private final QueryText queryText;
    private final AlertStatesChunksDao chunks;

    public YdbAlertsStateFilesDao(String path, TableClient tableClient, MetricRegistry registry) {
        this.tablePath = path + "/StatesFiles";
        this.table = new FileTable(tableClient, tablePath);
        this.queryText = TEMPLATE.build(Collections.singletonMap("alert_state_files.table.path", tablePath));
        this.chunks = DaoMetricsProxy
            .of(new YdbAlertStateChunksDao(path, tableClient), AlertStatesChunksDao.class, registry);
    }

    @Override
    public CompletableFuture<?> upload(String projectId, String fileId, long createdAt, List<TPersistAlertState> states) {
        try {
            var payload = StatesCompressor.compress(states);
            var file = new File();
            file.projectId = projectId;
            file.fileId = fileId;
            file.chunkCount = (int) Math.floor((double) payload.getCompressedBytesSize() / chunks.getChunkSize()) + 1;
            file.rawBytesSize = payload.getRawBytesSize();
            file.compressedBytesSize = payload.getCompressedBytesSize();
            file.time = createdAt;

            // Insert a huge state into one kikimr transaction too expensive, so we insert
            // each chunk of huge state into own transaction. To avoid junk into chunks table,
            // firstly inserted file metadata - cleanup script can read this metadata and found
            // all junk chunks without full scan whole chunks table.
            String query = queryText.query("insert");
            Params params = table.toParams(file);
            ByteString bytes = UnsafeByteOperations.unsafeWrap(payload.getCompressed());
            return table.queryVoid(query, params)
                .thenCompose(ignore -> chunks.uploadChunks(projectId, fileId, bytes));
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public CompletableFuture<Void> download(String projectId, String fileId, Consumer<TPersistAlertState> consumer) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$projectId", utf8(projectId), "$fileId", utf8(fileId));
            return table.queryOne(query, params)
                .thenCompose(opt -> {
                    var file = opt.orElseThrow(() -> Status.NOT_FOUND.withDescription("Not found file with id " + fileId).asRuntimeException());
                    return chunks.downloadChunks(file.projectId, file.fileId, file.chunkCount)
                        .thenAccept(compressed -> decompress(consumer, file, compressed));
                });
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private void decompress(Consumer<TPersistAlertState> consumer, File file, ByteString compressed) {
        if (compressed.size() != file.compressedBytesSize) {
            throw Status.DATA_LOSS
                .withDescription("Expected read bytes " + file.compressedBytesSize + " but was read " + compressed.size())
                .asRuntimeException();
        }

        StatesCompressor.decompress(compressed, file.rawBytesSize, consumer);
    }

    @Override
    public CompletableFuture<?> deleteFile(String projectId, String fileId) {
        return chunks.deleteFileChunks(projectId, fileId)
            .thenCompose(ignore -> {
                String query = queryText.query("delete_file");
                Params params = Params.of("$projectId", utf8(projectId), "$fileId", utf8(fileId));
                return table.queryVoid(query, params);
            });
    }

    @Override
    public CompletableFuture<?> deleteProject(String projectId) {
        return chunks.deleteProject(projectId)
            .thenCompose(ignore -> {
                String query = queryText.query("delete_all");
                Params params = Params.of("$projectId", utf8(projectId));
                return table.queryVoid(query, params);
            });
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return chunks.createSchemaForTests()
            .thenCompose(ignore -> table.create());
    }

    /**
     * FILES TABLE
     */
    private static final class FileTable extends YdbTable<String, File> {
        FileTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("rawBytesSize", PrimitiveType.uint32())
                .addNullableColumn("compressedBytesSize", PrimitiveType.uint32())
                .addNullableColumn("chunkCount", PrimitiveType.uint32())
                .addNullableColumn("leaderSeqNo", PrimitiveType.uint64()) // unused
                .addNullableColumn("projectSeqNo", PrimitiveType.uint64()) // unused
                .addNullableColumn("time", PrimitiveType.int64())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected String getId(File record) {
            return record.fileId;
        }

        @Override
        protected Params toParams(File record) {
            return Params.create()
                .put("$projectId", utf8(record.projectId))
                .put("$fileId", utf8(record.fileId))
                .put("$rawBytesSize", uint32(record.rawBytesSize))
                .put("$compressedBytesSize", uint32(record.compressedBytesSize))
                .put("$chunkCount", uint32(record.chunkCount))
                .put("$time", int64(record.time));
        }

        @Override
        protected File mapFull(ResultSetReader r) {
            var file = new File();
            file.projectId = r.getColumn("projectId").getUtf8();
            file.fileId = r.getColumn("id").getUtf8();
            file.rawBytesSize = (int) r.getColumn("rawBytesSize").getUint32();
            file.compressedBytesSize = (int) r.getColumn("compressedBytesSize").getUint32();
            file.chunkCount = (int) r.getColumn("chunkCount").getUint32();
            file.time = r.getColumn("time").getInt64();
            return file;
        }

        @Override
        protected File mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }

    private static class File {
        String projectId;
        String fileId;
        int rawBytesSize;
        int compressedBytesSize;
        int chunkCount;
        long time;
    }
}
