package ru.yandex.solomon.alert.dao.ydb.entity;

import java.nio.file.Path;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.dao.AlertStatesDao;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.protobuf.TPersistAlertState;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbAlertsStatesDao implements AlertStatesDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbAlertsStatesDao.class);
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
        YdbTelegramDao.class,
        "alert_state_indexes",
        List.of(
            "delete_all",
            "find",
            "replace"
        ));

    private final String tablePath;
    private final IndexTable table;
    private final SchemeClient scheme;
    private final QueryText queryText;
    private final AlertStatesFilesDao files;

    public YdbAlertsStatesDao(String path, TableClient tableClient, SchemeClient schemeClient, YdbSchemaVersion version, MetricRegistry registry) {
        var root = path + "/Alerting/" + version.folderName();
        this.tablePath = root + "/StatesIndexes";
        this.table = new IndexTable(tableClient, tablePath);
        this.scheme = schemeClient;
        this.queryText = TEMPLATE.build(Collections.singletonMap("alert_state_indexes.table.path", tablePath));
        this.files = DaoMetricsProxy
            .of(new YdbAlertsStateFilesDao(root, tableClient, registry), AlertStatesFilesDao.class, registry);
    }

    @Override
    public CompletableFuture<?> createSchema(String projectId) {
        return completedFuture(null);
    }

    @Override
    public CompletableFuture<?> save(String projectId, Instant time, AssignmentSeqNo seqNo, List<TPersistAlertState> states) {
        Index index = new Index();
        index.projectId = projectId;
        index.leaderSeqNo = seqNo.getLeaderSeqNo();
        index.projectSeqNo = seqNo.getAssignSeqNo();
        index.time = time.toEpochMilli();
        index.fileId = UUID.randomUUID().toString();

        return findIndex(projectId)
            .thenCompose(prev -> {
                if (prev.isPresent() && prev.get().compareTo(index) >= 0) {
                    logger.debug("{}: skip save state {} because obsoleted, new {}", projectId, seqNo, prev.get().getSeqNo());
                    return completedFuture(null);
                }

                return updateState(index, states);
            });
    }

    @Override
    public CompletableFuture<Void> find(String projectId, Consumer<TPersistAlertState> consumer) {
        return findIndex(projectId)
            .thenCompose(opt -> {
                if (opt.isEmpty()) {
                    return completedFuture(null);
                }

                var index = opt.get();
                return files.download(projectId, index.fileId, consumer);
            });
    }

    @Override
    public CompletableFuture<?> deleteProject(String projectId) {
        return files.deleteProject(projectId)
            .thenCompose(ignore -> {
                String query = queryText.query("delete_all");
                Params params = Params.of("$projectId", utf8(projectId));
                return table.queryVoid(query, params);
            });
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return scheme.makeDirectories(Path.of(tablePath).getParent().toString())
            .thenCompose(ignore -> files.createSchemaForTests())
            .thenCompose(ignore -> table.create());
    }

    private CompletableFuture<Optional<Index>> replaceIndex(Index index) {
        try {
            String query = queryText.query("replace");
            Params params = table.toParams(index);
            return table.queryOne(query, params);
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private CompletableFuture<Optional<Index>> findIndex(String projectId) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$projectId", utf8(projectId));
            return table.queryOne(query, params);
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private CompletableFuture<?> updateState(Index index, List<TPersistAlertState> states) {
        return files.upload(index.projectId, index.fileId, index.time, states)
            .thenCompose(ignore -> replaceIndex(index))
            .handle((prev, e) -> {
                if (e != null) {
                    logger.error("Save snapshot {} failed", index, e);
                    return files.deleteFile(index.projectId, index.fileId);
                } else if (prev.isEmpty()) {
                    return completedFuture(null);
                }

                var obsoleteFile = min(prev.get(), index).fileId;
                return files.deleteFile(index.projectId, obsoleteFile);
            })
            .thenCompose(future -> future);
    }

    private Index min(Index left, Index right) {
        if (left.compareTo(right) <= 0) {
            return left;
        }

        return right;
    }

    /**
     * INDEX TABLE
     */
    private static final class IndexTable extends YdbTable<String, Index> {
        IndexTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("fileId", PrimitiveType.utf8())
                .addNullableColumn("leaderSeqNo", PrimitiveType.uint64())
                .addNullableColumn("projectSeqNo", PrimitiveType.uint64())
                .addNullableColumn("time", PrimitiveType.int64())
                .setPrimaryKeys("projectId")
                .build();
        }

        @Override
        protected String getId(Index record) {
            return record.projectId;
        }

        @Override
        protected Params toParams(Index record) {
            return Params.create()
                .put("$projectId", utf8(record.projectId))
                .put("$fileId", utf8(record.fileId))
                .put("$leaderSeqNo", uint64(record.leaderSeqNo))
                .put("$projectSeqNo", uint64(record.projectSeqNo))
                .put("$time", int64(record.time));
        }

        @Override
        protected Index mapFull(ResultSetReader r) {
            var index = new Index();
            index.projectId = r.getColumn("projectId").getUtf8();
            index.fileId = r.getColumn("fileId").getUtf8();
            index.leaderSeqNo = r.getColumn("leaderSeqNo").getUint64();
            index.projectSeqNo = r.getColumn("projectSeqNo").getUint64();
            index.time = r.getColumn("time").getInt64();
            return index;
        }

        @Override
        protected Index mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }

    private static class Index implements Comparable<Index> {
        String projectId;
        String fileId;
        long leaderSeqNo;
        long projectSeqNo;
        long time;

        AssignmentSeqNo getSeqNo() {
            return new AssignmentSeqNo(leaderSeqNo, projectSeqNo);
        }

        @Override
        public int compareTo(Index record) {
            int compare = AssignmentSeqNo.compareLeaderSeqNo(leaderSeqNo, record.leaderSeqNo);
            if (compare == 0) {
                compare = AssignmentSeqNo.compareAssignSeqNo(projectSeqNo, record.projectSeqNo);
            }
            if (compare == 0) {
                compare = Long.compare(time, record.time);
            }

            return compare;
        }

        @Override
        public String toString() {
            return "Index{" +
                "projectId='" + projectId + '\'' +
                ", fileId='" + fileId + '\'' +
                ", leaderSeqNo=" + leaderSeqNo +
                ", projectSeqNo=" + projectSeqNo +
                ", time=" + time +
                '}';
        }
    }
}
