package ru.yandex.solomon.alert.dao.ydb.entity;

import java.nio.file.Path;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.alert.dao.MutesDao;
import ru.yandex.solomon.alert.dao.codec.MuteCodec;
import ru.yandex.solomon.alert.dao.codec.MuteRecord;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.uint64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Ivan Tsybulin
 */
public class YdbMutesDao implements MutesDao {
    private static final int SELECT_LIMIT = 1000;
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
        YdbMutesDao.class,
        "mutes",
        List.of(
            "delete_all",
            "delete_one",
            "insert",
            "update",
            "select_all_limited"
        ));

    private final String tablePath;
    private final MutesTable table;
    private final SchemeClient scheme;
    private final QueryText queryText;

    public YdbMutesDao(String path, TableClient tableClient, SchemeClient schemeClient, YdbSchemaVersion version, ObjectMapper objectMapper) {
        this.tablePath = path + "/Alerting/" + version.folderName() + "/Mutes";
        this.table = new MutesTable(tableClient, tablePath, objectMapper);
        this.scheme = schemeClient;
        this.queryText = TEMPLATE.build(Map.of("mutes.table.path", tablePath, "limit", SELECT_LIMIT));
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return scheme.makeDirectories(Path.of(tablePath).getParent().toString())
            .thenAccept(status -> status.expect("parent directories success created"))
            .thenCompose(ignore -> scheme.describePath(tablePath))
            .thenCompose(exist -> !exist.isSuccess()
                ? table.create()
                : completedFuture(null));
    }

    @Override
    public CompletableFuture<?> createSchema(String projectId) {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Optional<Mute>> insert(Mute channel, IdempotentOperation op) {
        return table.updateOne(queryText.query("insert"), channel);
    }

    @Override
    public CompletableFuture<Optional<Mute>> update(Mute channel, IdempotentOperation op) {
        return table.updateOne(queryText.query("update"), channel);
    }

    @Override
    public CompletableFuture<Void> deleteById(String projectId, String id, IdempotentOperation op) {
        try {
            String query = queryText.query("delete_one");
            Params params = Params.of("$id", utf8(id), "$projectId", utf8(projectId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteProject(String projectId) {
        try {
            String query = queryText.query("delete_all");
            Params params = Params.of("$projectId", utf8(projectId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> find(String projectId, Consumer<Mute> consumer) {
        return findAllBySelect(projectId, consumer)
                .thenCompose(truncated -> {
                    if (!truncated) {
                        return completedFuture(null);
                    }

                    return findAllByReadTable(projectId, consumer);
                });
    }

    @Override
    public CompletableFuture<Set<String>> findProjects() {
        var settings = ReadTableSettings.newBuilder()
                .timeout(1, TimeUnit.MINUTES)
                .column("projectId")
                .build();

        Set<String> result = new HashSet<>();
        Predicate<ResultSetReader> predicate = rs -> {
            result.add(rs.getColumn(0).getUtf8());
            return false;
        };
        return table.queryAll(predicate, settings).thenApply(ignore -> result);
    }

    private CompletableFuture<Boolean> findAllBySelect(String projectId, Consumer<Mute> consumer) {
        try {
            String query = queryText.query("select_all_limited");
            Params params = Params.of("$projectId", utf8(projectId));
            return table.executeAndExpectSuccess(query, params)
                    .thenApply(result -> {
                        var resultSet = result.getResultSet(0);
                        if (resultSet.getRowCount() >= SELECT_LIMIT || resultSet.isTruncated()) {
                            return Boolean.TRUE;
                        }

                        while (resultSet.next()) {
                            consumer.accept(table.mapFull(resultSet));
                        }

                        return Boolean.FALSE;
                    });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> findAllByReadTable(String projectId, Consumer<Mute> consumer) {
        var settings = ReadTableSettings.newBuilder()
                .timeout(1, TimeUnit.MINUTES)
                .fromKeyInclusive(utf8(projectId))
                .toKeyInclusive(utf8(projectId))
                .build();

        return table.queryAll(ignore -> true, settings)
                .thenAccept(alerts -> alerts.forEach(consumer));
    }

    private static final class MutesTable extends YdbTable<String, Mute> {
        private final MuteCodec codec;

        MutesTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.codec = new MuteCodec(objectMapper);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("ticketId", PrimitiveType.utf8())
                .addNullableColumn("fromTime", PrimitiveType.int64())
                .addNullableColumn("toTime", PrimitiveType.int64())
                .addNullableColumn("ttlBase", PrimitiveType.uint64())
                .addNullableColumn("type", PrimitiveType.int32())
                .addNullableColumn("config", PrimitiveType.utf8())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("version", PrimitiveType.uint32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected String getId(Mute channel) {
            return channel.getId();
        }

        @Override
        protected Params toParams(Mute channel) {
            var record = codec.encode(channel);
            return Params.create()
                .put("$projectId", utf8(record.projectId))
                .put("$id", utf8(record.id))
                .put("$folderId", utf8(record.folderId))
                .put("$name", utf8(record.name))
                .put("$description", utf8(record.description))
                .put("$ticketId", utf8(record.ticketId))
                .put("$fromTime", int64(record.from))
                .put("$toTime", int64(record.to))
                .put("$ttlBase", uint64(record.ttlBase))
                .put("$type", int32(record.type))
                .put("$config", utf8(record.config))
                .put("$createdBy", utf8(record.createdBy))
                .put("$updatedBy", utf8(record.updatedBy))
                .put("$createdAt", int64(record.createdAt))
                .put("$updatedAt", int64(record.updatedAt))
                .put("$version", uint32(record.version));
        }

        @Override
        protected Mute mapFull(ResultSetReader r) {
            var record = new MuteRecord();
            record.projectId = r.getColumn("projectId").getUtf8();
            record.id = r.getColumn("id").getUtf8();
            record.folderId = r.getColumn("folderId").getUtf8();
            record.name = r.getColumn("name").getUtf8();
            record.description = r.getColumn("description").getUtf8();
            record.ticketId = r.getColumn("ticketId").getUtf8();
            record.from = r.getColumn("fromTime").getInt64();
            record.to = r.getColumn("toTime").getInt64();
            record.ttlBase = r.getColumn("ttlBase").getUint64();
            record.type = r.getColumn("type").getInt32();
            record.config = r.getColumn("config").getUtf8();
            record.createdBy = r.getColumn("createdBy").getUtf8();
            record.updatedBy = r.getColumn("updatedBy").getUtf8();
            record.createdAt = r.getColumn("createdAt").getInt64();
            record.updatedAt = r.getColumn("updatedAt").getInt64();
            record.version = (int) r.getColumn("version").getUint32();
            return codec.decode(record);
        }

        @Override
        protected Mute mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
