package ru.yandex.solomon.alert.dao.ydb.entity;

import java.nio.file.Path;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AlterTableSettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.dao.NotificationsDao;
import ru.yandex.solomon.alert.dao.codec.NotificationCodec;
import ru.yandex.solomon.alert.dao.codec.NotificationRecord;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.setFromTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsv;
import static ru.yandex.solomon.core.db.dao.kikimr.KikimrDaoSupport.toTsvRegularExpression;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbNotificationsDao implements NotificationsDao {
    private static final int SELECT_LIMIT = 1000;
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
        YdbNotificationsDao.class,
        "notifications",
        List.of(
            "delete_all",
            "delete_one",
            "delete_one_with_validations",
            "insert",
            "update",
            "update_with_validations",
            "select_all_limited"
        ));

    private final String tablePath;
    private final NotificationsTable table;
    private final SchemeClient scheme;
    private final QueryText queryText;

    public YdbNotificationsDao(String path, TableClient tableClient, SchemeClient schemeClient, YdbSchemaVersion version, ObjectMapper objectMapper) {
        this.tablePath = path + "/Alerting/" + version.folderName() + "/Notifications";
        this.table = new NotificationsTable(tableClient, tablePath, objectMapper);
        this.scheme = schemeClient;
        this.queryText = TEMPLATE.build(Map.of("notifications.table.path", tablePath, "limit", SELECT_LIMIT));
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return scheme.makeDirectories(Path.of(tablePath).getParent().toString())
            .thenAccept(status -> status.expect("parent directories success created"))
            .thenCompose(ignore -> scheme.describePath(tablePath))
            .thenCompose(exist -> !exist.isSuccess()
                ? table.create()
                : completedFuture(null));
    }

    @Override
    public CompletableFuture<?> createSchema(String projectId) {
        return CompletableFuture.completedFuture(null);
    }

    public CompletableFuture<Void> migrateSchema() {
        return table.retryCtx().supplyResult(session -> session.describeTable(tablePath))
                .thenCompose(result -> {
                    if (!result.isSuccess()) {
                        return createSchemaForTests()
                                .thenApply(o -> null);
                    }
                    return alterTable(result.expect("unable describe " + tablePath));
                });
    }

    private CompletableFuture<Void> alterTable(TableDescription tableDescription) {
        for (var column : tableDescription.getColumns()) {
            if ("defaultForSeverity".equals(column.getName())) {
                return completedFuture(null);
            }
        }

        var settings = new AlterTableSettings();
        settings.addColumn("defaultForSeverity", PrimitiveType.utf8().makeOptional());
        return table.retryCtx().supplyStatus(session -> session.alterTable(tablePath, settings))
                .thenAccept(status -> status.expect("unable alter " + tablePath));
    }

    @Override
    public CompletableFuture<Optional<Notification>> insert(Notification channel, IdempotentOperation op) {
        return table.updateOne(queryText.query("insert"), channel);
    }

    @Override
    public CompletableFuture<Optional<Notification>> update(Notification channel, IdempotentOperation op) {
        return table.updateOne(queryText.query("update"), channel);
    }

    @Override
    public CompletableFuture<Optional<Notification>> updateWithValidations(Notification entity, IdempotentOperation op, Set<AlertSeverity> validateSeverities) {
        try {
            if (validateSeverities.isEmpty()) {
                return update(entity, op);
            }

            String query = queryText.query("update_with_validations");
            var params = table.toParams(entity)
                    .put("$validateDisaster", utf8(getSeverityRegexp(validateSeverities, AlertSeverity.DISASTER)))
                    .put("$validateInfo", utf8(getSeverityRegexp(validateSeverities, AlertSeverity.INFO)))
                    .put("$validateCritical", utf8(getSeverityRegexp(validateSeverities, AlertSeverity.CRITICAL)));
            return table.updateOne(query, params)
                    .exceptionally(this::handleEnsureException);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteById(String projectId, String id, IdempotentOperation op) {
        try {
            String query = queryText.query("delete_one");
            Params params = Params.of("$id", utf8(id), "$projectId", utf8(projectId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteByIdWithValidations(String projectId, String id, IdempotentOperation op, Set<AlertSeverity> validateSeverities) {
        try {
            if (validateSeverities.isEmpty()) {
                return deleteById(projectId, id, op);
            }

            String query = queryText.query("delete_one_with_validations");
            Params params = Params.of("$id", utf8(id),
                    "$projectId", utf8(projectId),
                    "$validateDisaster", utf8(getSeverityRegexp(validateSeverities, AlertSeverity.DISASTER)),
                    "$validateInfo", utf8(getSeverityRegexp(validateSeverities, AlertSeverity.INFO)),
                    "$validateCritical", utf8(getSeverityRegexp(validateSeverities, AlertSeverity.CRITICAL)));
            return table.queryVoid(query, params)
                    .exceptionally(this::handleEnsureException);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private <T> T handleEnsureException(Throwable throwable) {
        Throwable cause = CompletableFutures.unwrapCompletionException(throwable);
        if (cause instanceof UnexpectedResultException) {
            for (AlertSeverity value : AlertSeverity.values()) {
                if (cause.getMessage().contains("Try to delete last default channel for " + value + " severity")) {
                    throw new IllegalArgumentException("Try to delete last default channel for " + value + " severity");
                }
            }
        }
        throw new RuntimeException(throwable);
    }

    private String getSeverityRegexp(Set<AlertSeverity> validateSeverities, AlertSeverity severity) {
        if (validateSeverities.contains(severity)) {
            return toTsvRegularExpression(severity.name());
        }
        return "";
    }

    @Override
    public CompletableFuture<Void> deleteProject(String projectId) {
        try {
            String query = queryText.query("delete_all");
            Params params = Params.of("$projectId", utf8(projectId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> find(String projectId, Consumer<Notification> consumer) {
        return findAllBySelect(projectId, consumer)
                .thenCompose(truncated -> {
                    if (!truncated) {
                        return completedFuture(null);
                    }

                    return findAllByReadTable(projectId, consumer);
                });
    }

    @Override
    public CompletableFuture<Set<String>> findProjects() {
        var settings = ReadTableSettings.newBuilder()
                .timeout(1, TimeUnit.MINUTES)
                .column("projectId")
                .build();

        Set<String> result = new HashSet<>();
        Predicate<ResultSetReader> predicate = rs -> {
            result.add(rs.getColumn(0).getUtf8());
            return false;
        };
        return table.queryAll(predicate, settings).thenApply(ignore -> result);
    }

    private CompletableFuture<Boolean> findAllBySelect(String projectId, Consumer<Notification> consumer) {
        try {
            String query = queryText.query("select_all_limited");
            Params params = Params.of("$projectId", utf8(projectId));
            return table.executeAndExpectSuccess(query, params)
                    .thenApply(result -> {
                        var resultSet = result.getResultSet(0);
                        if (resultSet.getRowCount() >= SELECT_LIMIT || resultSet.isTruncated()) {
                            return Boolean.TRUE;
                        }

                        while (resultSet.next()) {
                            consumer.accept(table.mapFull(resultSet));
                        }

                        return Boolean.FALSE;
                    });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> findAllByReadTable(String projectId, Consumer<Notification> consumer) {
        var settings = ReadTableSettings.newBuilder()
                .timeout(1, TimeUnit.MINUTES)
                .fromKeyInclusive(utf8(projectId))
                .toKeyInclusive(utf8(projectId))
                .build();

        return table.queryAll(ignore -> true, settings)
                .thenAccept(alerts -> alerts.forEach(consumer));
    }

    /**
     * ALERTS TABLE
     */
    private static final class NotificationsTable extends YdbTable<String, Notification> {
        private final NotificationCodec codec;

        NotificationsTable(TableClient tableClient, String path, ObjectMapper objectMapper) {
            super(tableClient, path);
            this.codec = new NotificationCodec(objectMapper);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("folderId", PrimitiveType.utf8())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("defaultForSeverity", PrimitiveType.utf8())
                .addNullableColumn("type", PrimitiveType.int32())
                .addNullableColumn("subscribeOn", PrimitiveType.uint32())
                .addNullableColumn("repeatNotifyDelay", PrimitiveType.int64())
                .addNullableColumn("config", PrimitiveType.utf8())
                .addNullableColumn("labels", PrimitiveType.utf8())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedBy", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.int64())
                .addNullableColumn("updatedAt", PrimitiveType.int64())
                .addNullableColumn("defaultForProject", PrimitiveType.bool())
                .addNullableColumn("version", PrimitiveType.uint32())
                .setPrimaryKeys("projectId", "id")
                .build();
        }

        @Override
        protected String getId(Notification channel) {
            return channel.getId();
        }

        @Override
        protected Params toParams(Notification channel) {
            var record = codec.encode(channel);
            return Params.create()
                .put("$projectId", utf8(record.projectId))
                .put("$id", utf8(record.id))
                .put("$folderId", utf8(record.folderId))
                .put("$name", utf8(record.name))
                .put("$description", utf8(record.description))
                .put("$defaultForSeverity", utf8(toTsv(record.defaultForSeverity)))
                .put("$type", int32(record.type))
                .put("$subscribeOn", uint32(record.subscribeOn))
                .put("$repeatNotifyDelay", int64(record.repeatNotifyDelay))
                .put("$labels", utf8(record.labels))
                .put("$config", utf8(record.config))
                .put("$createdBy", utf8(record.createdBy))
                .put("$updatedBy", utf8(record.updatedBy))
                .put("$createdAt", int64(record.createdAt))
                .put("$updatedAt", int64(record.updatedAt))
                .put("$defaultForProject", bool(record.defaultForProject))
                .put("$version", uint32(record.version));
        }

        @Override
        protected Notification mapFull(ResultSetReader r) {
            var record = new NotificationRecord();
            record.projectId = r.getColumn("projectId").getUtf8();
            record.id = r.getColumn("id").getUtf8();
            record.folderId = r.getColumn("folderId").getUtf8();
            record.name = r.getColumn("name").getUtf8();
            record.description = r.getColumn("description").getUtf8();
            record.type = r.getColumn("type").getInt32();
            record.subscribeOn = (int) r.getColumn("subscribeOn").getUint32();
            record.repeatNotifyDelay = r.getColumn("repeatNotifyDelay").getInt64();
            record.config = r.getColumn("config").getUtf8();
            record.labels = r.getColumn("labels").getUtf8();
            record.createdBy = r.getColumn("createdBy").getUtf8();
            record.updatedBy = r.getColumn("updatedBy").getUtf8();
            record.createdAt = r.getColumn("createdAt").getInt64();
            record.updatedAt = r.getColumn("updatedAt").getInt64();
            record.version = (int) r.getColumn("version").getUint32();
            record.defaultForProject = r.getColumn("defaultForProject").getBool();
            record.defaultForSeverity = setFromTsv(r.getColumn("defaultForSeverity").getUtf8());
            return codec.decode(record);
        }

        @Override
        protected Notification mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
