package ru.yandex.solomon.alert.dao.ydb.entity;

import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.alert.dao.ShardsDao;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbShardsDao implements ShardsDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
            YdbTelegramDao.class,
            "shards",
            List.of("delete", "insert"));

    private final String tablePath;
    private final ShardsTable table;
    private final SchemeClient scheme;
    private final QueryText queryText;

    public YdbShardsDao(String path, TableClient tableClient, SchemeClient schemeClient, YdbSchemaVersion version) {
        this.tablePath = path + "/Alerting/" + version.folderName() + "/Shards";
        this.table = new ShardsTable(tableClient, tablePath);
        this.scheme = schemeClient;
        this.queryText = TEMPLATE.build(Collections.singletonMap("shards.table.path", tablePath));
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return scheme.makeDirectories(Path.of(tablePath).getParent().toString())
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(ignore -> scheme.describePath(tablePath))
                .thenCompose(exist -> !exist.isSuccess()
                        ? table.create()
                        : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> insert(String shardId) {
        try {
            String query = queryText.query("insert");
            Params params = Params.of("$projectId", utf8(shardId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> delete(String shardId) {
        try {
            String query = queryText.query("delete");
            Params params = Params.of("$projectId", utf8(shardId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<String>> findAll() {
        return table.queryAll();
    }

    /**
     * SHARDS TABLE
     */
    private static final class ShardsTable extends YdbTable<String, String> {

        ShardsTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                    .addNullableColumn("projectId", PrimitiveType.utf8())
                    .setPrimaryKeys("projectId")
                    .build();
        }

        @Override
        protected String getId(String record) {
            return record;
        }

        @Override
        protected Params toParams(String record) {
            return Params.create()
                    .put("$projectId", utf8(record));
        }

        @Override
        protected String mapFull(ResultSetReader r) {
            return r.getColumn("projectId").getUtf8();
        }

        @Override
        protected String mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
