package ru.yandex.solomon.alert.dao.ydb.entity;

import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.alert.dao.TelegramDao;
import ru.yandex.solomon.alert.dao.TelegramRecord;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbTelegramDao implements TelegramDao {
    private static final QueryTemplate TEMPLATE = new QueryTemplate(
        YdbTelegramDao.class,
        "telegram",
        List.of(
            "delete_one",
            "find",
            "insert"
        ));

    private final String tablePath;
    private final TelegramTable table;
    private final SchemeClient scheme;
    private final QueryText queryText;

    public YdbTelegramDao(String path, TableClient tableClient, SchemeClient schemeClient, YdbSchemaVersion version) {
        this.tablePath = path + "/Alerting/" + version.folderName() + "/Telegram";
        this.table = new TelegramTable(tableClient, tablePath);
        this.scheme = schemeClient;
        this.queryText = TEMPLATE.build(Collections.singletonMap("telegram.table.path", tablePath));
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return scheme.makeDirectories(Path.of(tablePath).getParent().toString())
            .thenAccept(status -> status.expect("parent directories success created"))
            .thenCompose(ignore -> scheme.describePath(tablePath))
            .thenCompose(exist -> !exist.isSuccess()
                ? table.create()
                : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> migrate() {
        return completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> upsert(TelegramRecord record) {
        try {
            String query = queryText.query("insert");
            Params params = table.toParams(record);
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<TelegramRecord>> get(long chatId) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$chatId", int64(chatId));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteById(long chatId) {
        try {
            String query = queryText.query("delete_one");
            Params params = Params.of("$chatId", int64(chatId));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<TelegramRecord>> findAll() {
        return table.queryAll();
    }

    /**
     * ALERTS TABLE
     */
    private static final class TelegramTable extends YdbTable<Long, TelegramRecord> {

        TelegramTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("chatId", PrimitiveType.int64())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("group", PrimitiveType.bool())
                .setPrimaryKeys("chatId")
                .build();
        }

        @Override
        protected Long getId(TelegramRecord record) {
            return record.getChatId();
        }

        @Override
        protected Params toParams(TelegramRecord record) {
            return Params.create()
                .put("$chatId", int64(record.getChatId()))
                .put("$name", utf8(record.getName()))
                .put("$group", bool(record.isGroup()));
        }

        @Override
        protected TelegramRecord mapFull(ResultSetReader r) {
            long chatId = r.getColumn("chatId").getInt64();
            String name = r.getColumn("name").getUtf8();
            boolean group = r.getColumn("group").getBool();
            return new TelegramRecord(chatId, name, group);
        }

        @Override
        protected TelegramRecord mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }
}
