package ru.yandex.solomon.alert.dao.ydb.entity;

import java.io.IOException;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.dao.TelegramEventContext;
import ru.yandex.solomon.alert.dao.TelegramEventRecord;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.notification.channel.telegram.EventAppearance;
import ru.yandex.solomon.core.db.dao.kikimr.QueryTemplate;
import ru.yandex.solomon.core.db.dao.kikimr.QueryText;
import ru.yandex.solomon.ydb.YdbTable;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbTelegramEventsDao implements TelegramEventsDao {
    private static final StructType KEY_TYPE = StructType.of("id", PrimitiveType.utf8());
    private static final ListType KEY_LIST_TYPE = ListType.of(KEY_TYPE);

    private static final QueryTemplate TEMPLATE = new QueryTemplate(
        YdbTelegramEventsDao.class,
        "telegram_events",
        List.of(
            "delete_older",
            "find",
            "insert",
            "update_context"
        ));

    private static final ObjectMapper mapper = new ObjectMapper();
    private static final Logger logger = LoggerFactory.getLogger(TelegramEventsDao.class);

    private final String tablePath;
    private final TelegramEventsTable table;
    private final SchemeClient scheme;
    private final QueryText queryText;
    private final Rate deleteRowsCounter = MetricRegistry.root().rate("telegramEvents.deleteRows");

    public YdbTelegramEventsDao(String path, TableClient tableClient, SchemeClient schemeClient, YdbSchemaVersion version) {
        this.tablePath = path + "/Alerting/" + version.folderName() + "/TelegramEvents";
        this.table = new TelegramEventsTable(tableClient, tablePath);
        this.scheme = schemeClient;
        this.queryText = TEMPLATE.build(Collections.singletonMap("telegram_events.table.path", tablePath));
    }

    @Override
    public CompletableFuture<?> createSchemaForTests() {
        return scheme.makeDirectories(Path.of(tablePath).getParent().toString())
            .thenAccept(status -> status.expect("parent directories success created"))
            .thenCompose(ignore -> scheme.describePath(tablePath))
            .thenCompose(exist -> !exist.isSuccess()
                ? table.create()
                : completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> insert(TelegramEventRecord record) {
        try {
            String query = queryText.query("insert");
            Params params = table.toParams(record);
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<TelegramEventRecord>> find(String id) {
        try {
            String query = queryText.query("find");
            Params params = Params.of("$id", utf8(id));
            return table.queryOne(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Void> deleteOlderThan(Instant instant) {
        long deleteRowsOlderThan = instant.toEpochMilli();
        ReadTableSettings settings = ReadTableSettings.newBuilder()
            .orderedRead(true)
            .timeout(1, TimeUnit.MINUTES)
            .columns("id", "createdAt")
            .build();
        return table.queryAll(reader -> {
            long createdAt = reader.getColumn("createdAt").getUint64();
            return createdAt < deleteRowsOlderThan;
        }, settings).thenCompose(records -> {
            if (records.isEmpty()) {
                return completedFuture(null);
            }

            var partitions = Lists.partition(records, 1000);
            CompletableFuture<Void> future = null;
            for (var partition : partitions) {
                if (future == null) {
                    future = deleteEvents(partition);
                } else {
                    future.thenCompose(ignore -> deleteEvents(partition));
                }
            }
            return future;
        });
    }

    @Override
    public CompletableFuture<Void> updateContext(TelegramEventRecord record) {
        try {
            String query = queryText.query("update_context");
            Params params = Params.create()
                    .put("$id", utf8(record.getId()))
                    .put("$context", utf8(toJson(record.getContext())));
            return table.queryVoid(query, params);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> deleteEvents(List<TelegramEventRecord> records) {
        var ids = records.stream()
            .map(record -> KEY_TYPE.newValueUnsafe(PrimitiveValue.utf8(record.getId())))
            .collect(Collectors.collectingAndThen(Collectors.toList(), KEY_LIST_TYPE::newValue));
        String query = queryText.query("delete_older");
        Params params = Params.of("$Keys", ids);
        int size = records.size();
        return table.queryVoid(query, params)
            .thenAccept(ignore -> deleteRowsCounter.add(size));
    }

    /**
     * TELEGRAM EVENTS TABLE
     */
    private static final class TelegramEventsTable extends YdbTable<String, TelegramEventRecord> {

        TelegramEventsTable(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.uint64())
                .addNullableColumn("messageId", PrimitiveType.int64())
                .addNullableColumn("projectId", PrimitiveType.utf8())
                .addNullableColumn("alertId", PrimitiveType.utf8())
                .addNullableColumn("subAlertId", PrimitiveType.utf8())
                .addNullableColumn("evaluatedAt", PrimitiveType.uint64())
                .addNullableColumn("alertStatusCode", PrimitiveType.int32())
                .addNullableColumn("muteId", PrimitiveType.utf8())
                .addNullableColumn("context", PrimitiveType.utf8())
                .addNullableColumn("eventAppearance", PrimitiveType.int32())
                .setPrimaryKeys("id")
                .build();
        }

        @Override
        protected String getId(TelegramEventRecord record) {
            return record.getId();
        }

        @Override
        protected Params toParams(TelegramEventRecord record) {
            return Params.create()
                .put("$id", utf8(record.getId()))
                .put("$createdAt", uint64(record.getCreatedAt()))
                .put("$evaluatedAt", uint64(record.getEvaluatedAt()))
                .put("$projectId", utf8(record.getProjectId()))
                .put("$alertId", utf8(record.getAlertId()))
                .put("$subAlertId", utf8(record.getSubAlertId()))
                .put("$messageId", int64(record.getMessageId()))
                .put("$alertStatusCode", int32(record.getAlertStatusCode().getNumber()))
                .put("$muteId", utf8(record.getMuteId()))
                .put("$context", utf8(toJson(record.getContext())))
                .put("$eventAppearance", int32(record.getEventAppearance().getNumber()))
                ;
        }

        @Override
        protected TelegramEventRecord mapFull(ResultSetReader r) {
            var id = r.getColumn("id").getUtf8();
            var createdAt = r.getColumn("createdAt").getUint64();
            if (r.getColumnCount() == 2) {
                return TelegramEventRecord.makePartial(id, createdAt);
            }

            var evaluatedAt = r.getColumn("evaluatedAt").getUint64();
            var messageId = r.getColumn("messageId").getInt64();
            var projectId = r.getColumn("projectId").getUtf8();
            var alertId = r.getColumn("alertId").getUtf8();
            var subAlertId = r.getColumn("subAlertId").getUtf8();
            var alertStatusCode = r.getColumn("alertStatusCode").getInt32();
            var eventAppearance = r.getColumn("eventAppearance").getInt32();
            var muteId = r.getColumn("muteId").getUtf8();
            var context = r.getColumn("context").getUtf8();

            return new TelegramEventRecord(id, createdAt, messageId, projectId,
                    alertId, subAlertId, evaluatedAt, EvaluationStatus.Code.forNumber(alertStatusCode),
                    muteId,
                    fromJson(context),
                    EventAppearance.forNumber(eventAppearance));
        }

        @Override
        protected TelegramEventRecord mapPartial(ResultSetReader r) {
            return mapFull(r);
        }
    }

    private static String toJson(TelegramEventContext context) {
        try {
            return mapper.writeValueAsString(context);
        } catch (JsonProcessingException e) {
            logger.error("Exception while serializing event context {}", context, e);
            return "";
        }
    }

    private static TelegramEventContext fromJson(String context) {
        if (context.isEmpty()) {
            return new TelegramEventContext();
        }
        try {
            return mapper.readValue(context, TelegramEventContext.class);
        } catch (IOException e) {
            logger.error("Exception while deserializing event context {}", context, e);
            return new TelegramEventContext();
        }
    }
}
