package ru.yandex.solomon.alert.domain;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.EqualsExclude;

import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractAlert implements Alert {
    private final AlertKey key;
    private final String folderId;
    private final String name;
    private final String description;
    private final AlertState state;
    @EqualsExclude
    private final String createdBy;
    @EqualsExclude
    private final long createdAt;
    @EqualsExclude
    private final String updatedBy;
    @EqualsExclude
    private final long updatedAt;
    @EqualsExclude
    private final int version;
    private final List<String> groupByLabels;
    private final Map<String, ChannelConfig> notificationChannels;
    private final Set<String> escalations;
    private final Map<String, String> annotations;
    private final Map<String, String> serviceProviderAnnotations;
    private final Map<String, String> labels;
    private final Duration period;
    private final int delaySeconds;
    private final ResolvedEmptyPolicy resolvedEmptyPolicy;
    private final NoPointsPolicy noPointsPolicy;
    private final AlertSeverity severity;
    private final boolean obtainedFromTemplate;

    protected AbstractAlert(AbstractAlertBuilder<?, ?> builder) {
        this.key = AlertKeyInterner.I.intern(new AlertKey(builder.projectId, "", builder.id));
        this.folderId = Nullables.orEmpty(builder.folderId);
        this.name = Optional.ofNullable(builder.name).orElse("");
        this.description = Optional.ofNullable(builder.description).orElse("");
        this.state = Optional.ofNullable(builder.state).orElse(AlertState.ACTIVE);
        this.createdBy = Nullables.orEmpty(builder.createdBy);
        this.createdAt = builder.createdAt != 0 ? builder.createdAt : System.currentTimeMillis();
        this.updatedBy = Nullables.orEmpty(builder.updatedBy);
        this.updatedAt = builder.updatedAt != 0 ? builder.updatedAt : System.currentTimeMillis();
        this.version = builder.version;
        this.groupByLabels = builder.groupByLabels;
        this.notificationChannels = ImmutableMap.copyOf(builder.notificationChannels);
        this.escalations = ImmutableSet.copyOf(builder.escalations);
        this.annotations = ImmutableMap.copyOf(builder.annotations);
        this.serviceProviderAnnotations = ImmutableMap.copyOf(builder.serviceProviderAnnotations);
        this.labels = ImmutableMap.copyOf(builder.labels);
        this.period = builder.period != null ? builder.period : Duration.ofMinutes(5);
        this.delaySeconds = builder.delaySeconds;
        this.resolvedEmptyPolicy = builder.resolvedEmptyPolicy;
        this.noPointsPolicy = builder.noPointsPolicy;
        this.severity = builder.severity;
        this.obtainedFromTemplate = builder.obtainedFromTemplate;
    }

    @Override
    public String getId() {
        return key.getAlertId();
    }

    @Override
    public String getProjectId() {
        return key.getProjectId();
    }

    @Override
    public AlertKey getKey() {
        return key;
    }

    @Override
    public String getFolderId() {
        return folderId;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public AlertState getState() {
        return state;
    }

    @Override
    public String getCreatedBy() {
        return createdBy;
    }

    @Override
    public long getCreatedAt() {
        return createdAt;
    }

    @Override
    public String getUpdatedBy() {
        return updatedBy;
    }

    @Override
    public long getUpdatedAt() {
        return updatedAt;
    }

    @Override
    public int getVersion() {
        return version;
    }

    @Override
    public Map<String, ChannelConfig> getNotificationChannels() {
        return notificationChannels;
    }

    @Override
    public Set<String> getEscalations() {
        return escalations;
    }

    @Override
    public List<String> getGroupByLabels() {
        return groupByLabels;
    }

    @Override
    public Map<String, String> getAnnotations() {
        return annotations;
    }

    @Override
    public Map<String, String> getServiceProviderAnnotations() {
        return serviceProviderAnnotations;
    }

    @Override
    public Map<String, String> getLabels() {
        return labels;
    }

    @Override
    public Duration getPeriod() {
        return period;
    }

    @Override
    public int getDelaySeconds() {
        return delaySeconds;
    }

    @Override
    public ResolvedEmptyPolicy getResolvedEmptyPolicy() {
        return resolvedEmptyPolicy;
    }

    @Override
    public NoPointsPolicy getNoPointsPolicy() {
        return noPointsPolicy;
    }

    @Override
    public AlertSeverity getSeverity() {
        return severity;
    }

    public boolean isObtainedFromTemplate() {
        return obtainedFromTemplate;
    }

    @Override
    public int getMetricsLimit() {
        // TODO: fill from ru.yandex.solomon.config.protobuf.alert.ExecutorConfig::getMaxMetricsToLoadForAlertRule
        return 100;
    }

    @Override
    public final boolean equalContent(Alert alert) {
        return EqualsBuilder.reflectionEquals(this, alert);
    }

    @Override
    public abstract AlertType getAlertType();

    @Override
    public abstract AbstractAlertBuilder toBuilder();

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AbstractAlert that = (AbstractAlert) o;
        return createdAt == that.createdAt &&
                updatedAt == that.updatedAt &&
                version == that.version &&
                delaySeconds == that.delaySeconds &&
                key.equals(that.key) &&
                folderId.equals(that.folderId) &&
                name.equals(that.name) &&
                description.equals(that.description) &&
                state == that.state &&
                createdBy.equals(that.createdBy) &&
                updatedBy.equals(that.updatedBy) &&
                groupByLabels.equals(that.groupByLabels) &&
                notificationChannels.equals(that.notificationChannels) &&
                escalations.equals(that.escalations) &&
                annotations.equals(that.annotations) &&
                serviceProviderAnnotations.equals(that.serviceProviderAnnotations) &&
                labels.equals(that.labels) &&
                period.equals(that.period) &&
                resolvedEmptyPolicy == that.resolvedEmptyPolicy &&
                severity == that.severity &&
                obtainedFromTemplate == that.obtainedFromTemplate &&
                noPointsPolicy == that.noPointsPolicy;
    }

    @Override
    public int hashCode() {
        return Objects.hash(key, folderId, name, description, state, createdBy, createdAt, updatedBy, updatedAt,
                version, groupByLabels, notificationChannels, annotations, labels, period, delaySeconds,
                resolvedEmptyPolicy, noPointsPolicy, serviceProviderAnnotations, severity, obtainedFromTemplate, escalations);
    }

    @Override
    public String toString() {
        return "AbstractAlert{" +
            "key=" + key +
            ", folderId='" + folderId + '\'' +
            ", name='" + name + '\'' +
            ", state=" + state +
            ", createdBy='" + createdBy + '\'' +
            ", createdAt=" + createdAt +
            ", updatedBy='" + updatedBy + '\'' +
            ", updatedAt=" + updatedAt +
            ", version=" + version +
            ", groupByLabels=" + groupByLabels +
            ", notificationChannels=" + notificationChannels +
            ", escalations=" + escalations +
            ", annotations=" + annotations +
            ", serviceProviderAnnotations=" + serviceProviderAnnotations +
            ", labels=" + labels +
            ", period=" + period +
            ", delaySeconds=" + delaySeconds +
            ", resolvedEmptyPolicy=" + resolvedEmptyPolicy +
            ", noPointsPolicy=" + noPointsPolicy +
            ", severity=" + severity +
            ", obtainedFromTemplate=" + obtainedFromTemplate +
            '}';
    }
}
