package ru.yandex.solomon.alert.domain;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.solomon.alert.notification.ChannelConfigInterner;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public abstract class AbstractAlertBuilder<AlertType extends Alert, BuilderType extends AbstractAlertBuilder<AlertType, BuilderType>> {
    @Nullable
    protected String id;
    @Nullable
    String projectId;
    @Nullable
    String folderId;
    @Nullable
    String name;
    @Nullable
    String description;
    @Nullable
    AlertState state;
    @Nullable
    String createdBy;
    @Nullable
    long createdAt;
    @Nullable
    String updatedBy;
    @Nullable
    long updatedAt;
    int version = 0;
    Map<String, ChannelConfig> notificationChannels = Collections.emptyMap();
    Set<String> escalations = Collections.emptySet();
    List<String> groupByLabels = Collections.emptyList();
    Map<String, String> annotations = ImmutableMap.of();
    Map<String, String> serviceProviderAnnotations = ImmutableMap.of();
    Map<String, String> labels = ImmutableMap.of();
    @Nullable
    Duration period = null;
    int delaySeconds = 0;
    ResolvedEmptyPolicy resolvedEmptyPolicy = ResolvedEmptyPolicy.DEFAULT;
    NoPointsPolicy noPointsPolicy = NoPointsPolicy.DEFAULT;
    AlertSeverity severity = AlertSeverity.UNKNOWN;
    boolean obtainedFromTemplate = false;

    public AbstractAlertBuilder() {
    }

    public AbstractAlertBuilder(AlertType alert) {
        this.id = alert.getId();
        this.projectId = alert.getProjectId();
        this.folderId = alert.getFolderId();
        this.name = alert.getName();
        this.description = alert.getDescription();
        this.state = alert.getState();
        this.createdBy = alert.getCreatedBy();
        this.createdAt = alert.getCreatedAt();
        this.updatedBy = alert.getUpdatedBy();
        this.updatedAt = alert.getUpdatedAt();
        this.version = alert.getVersion();
        this.groupByLabels = alert.getGroupByLabels();
        this.notificationChannels = alert.getNotificationChannels();
        this.annotations = alert.getAnnotations();
        this.serviceProviderAnnotations = alert.getServiceProviderAnnotations();
        this.labels = alert.getLabels();
        this.period = alert.getPeriod();
        this.delaySeconds = alert.getDelaySeconds();
        this.resolvedEmptyPolicy = alert.getResolvedEmptyPolicy();
        this.noPointsPolicy = alert.getNoPointsPolicy();
        this.severity = alert.getSeverity();
        this.escalations = alert.getEscalations();
        this.obtainedFromTemplate = alert.isObtainedFromTemplate();
    }

    protected abstract BuilderType self();

    public BuilderType setId(String id) {
        this.id = id;
        return self();
    }

    public BuilderType setProjectId(String projectId) {
        this.projectId = StringInterner.I.intern(projectId);
        return self();
    }

    public BuilderType setFolderId(String folderId) {
        this.folderId = StringInterner.I.intern(folderId);
        return self();
    }

    public BuilderType setName(String name) {
        this.name = name;
        return self();
    }

    public BuilderType setDescription(String description) {
        this.description = description;
        return self();
    }

    public BuilderType setState(AlertState state) {
        this.state = state;
        return self();
    }

    public BuilderType setCreatedBy(String user) {
        this.createdBy = StringInterner.I.intern(user);
        return self();
    }

    public BuilderType setCreatedAt(long createdAt) {
        this.createdAt = Instant.ofEpochMilli(createdAt)
            .truncatedTo(ChronoUnit.SECONDS)
            .toEpochMilli();
        return self();
    }

    public BuilderType setUpdatedBy(String user) {
        this.updatedBy = StringInterner.I.intern(user);
        return self();
    }

    public BuilderType setUpdatedAt(long updatedAt) {
        this.updatedAt = Instant.ofEpochMilli(updatedAt)
            .truncatedTo(ChronoUnit.SECONDS)
            .toEpochMilli();
        return self();
    }

    public BuilderType setVersion(int version) {
        this.version = version;
        return self();
    }

    public BuilderType setNotificationChannels(Map<String, ChannelConfig> notificationChannels) {
        var map = ImmutableMap.<String, ChannelConfig>builder();
        for (var entry : notificationChannels.entrySet()) {
            map.put(StringInterner.I.intern(entry.getKey()), ChannelConfigInterner.I.intern(entry.getValue()));
        }
        this.notificationChannels = map.build();
        return self();
    }

    @Deprecated
    public BuilderType setNotificationChannels(Iterable<String> channelIds) {
        var map = ImmutableMap.<String, ChannelConfig>builder();
        for (var channelId : channelIds) {
            map.put(StringInterner.I.intern(channelId), ChannelConfig.EMPTY);
        }
        this.notificationChannels = map.build();
        return self();
    }

    // Used in tests only
    public BuilderType setNotificationChannel(String channelId) {
        this.notificationChannels = Map.of(StringInterner.I.intern(channelId), ChannelConfig.EMPTY);
        return self();
    }

    public BuilderType setEscalations(Set<String> escalations) {
        var set = ImmutableSet.<String>builder();
        for (var entry : escalations) {
            set.add(StringInterner.I.intern(entry));
        }
        this.escalations = set.build();
        return self();
    }

    // Used in tests only
    public BuilderType setEscalation(String escalation) {
        this.escalations = Set.of(StringInterner.I.intern(escalation));
        return self();
    }

    public BuilderType setGroupByLabels(Iterable<String> labels) {
        this.groupByLabels = StreamSupport.stream(labels.spliterator(), false)
            .map(StringInterner.I::intern)
            .collect(Collectors.toUnmodifiableList());
        return self();
    }

    public BuilderType setGroupByLabel(String label) {
        this.groupByLabels = List.of(StringInterner.I.intern(label));
        return self();
    }

    public BuilderType setAnnotations(Map<String, String> annotations) {
        var map = ImmutableMap.<String, String>builder();
        for (var entry : annotations.entrySet()) {
            map.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
        }
        this.annotations = map.build();
        return self();
    }

    public BuilderType addAnnotation(String key, String value) {
        this.annotations = ImmutableMap.<String, String>builder()
                .putAll(annotations)
                .put(StringInterner.I.intern(key), StringInterner.I.intern(value))
                .build();
        return self();
    }

    public BuilderType setServiceProviderAnnotations(Map<String, String> annotations) {
        var map = ImmutableMap.<String, String>builder();
        for (var entry : annotations.entrySet()) {
            map.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
        }
        this.serviceProviderAnnotations = map.build();
        return self();
    }

    public BuilderType addServiceProviderAnnotations(String key, String value) {
        this.serviceProviderAnnotations = ImmutableMap.<String, String>builder()
                .putAll(serviceProviderAnnotations)
                .put(StringInterner.I.intern(key), StringInterner.I.intern(value))
                .build();
        return self();
    }

    public BuilderType setLabels(Map<String, String> labels) {
        var map = ImmutableMap.<String, String>builder();
        for (var entry : labels.entrySet()) {
            map.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
        }
        this.labels = map.build();
        return self();
    }

    public BuilderType addLabel(String key, String value) {
        this.labels = ImmutableMap.<String, String>builder()
                .putAll(labels)
                .put(StringInterner.I.intern(key), StringInterner.I.intern(value))
                .build();
        return self();
    }

    public BuilderType setPeriod(Duration period) {
        this.period = period;
        return self();
    }

    public BuilderType setDelaySeconds(int delaySeconds) {
        this.delaySeconds = delaySeconds;
        return self();
    }

    public BuilderType setResolvedEmptyPolicy(ResolvedEmptyPolicy resolvedEmptyPolicy) {
        this.resolvedEmptyPolicy = resolvedEmptyPolicy;
        return self();
    }

    public BuilderType setNoPointsPolicy(NoPointsPolicy noPointsPolicy) {
        this.noPointsPolicy = noPointsPolicy;
        return self();
    }

    public BuilderType setSeverity(AlertSeverity severity) {
        this.severity = severity;
        return self();
    }

    public BuilderType setObtainedFromTemplate(boolean obtainedFromTemplate) {
        this.obtainedFromTemplate = obtainedFromTemplate;
        return self();
    }

    public abstract AlertType build();
}
