package ru.yandex.solomon.alert.domain;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.MoreObjects;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.unroll.MultiAlertUtils;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SubAlert implements Alert {
    private final AlertKey key;
    private final Alert parent;
    private final Labels groupKey;

    public SubAlert(Builder builder) {
        this.parent = requireNonNull(builder.parent);
        this.groupKey = builder.groupKey == null
                ? Labels.empty()
                : builder.groupKey;

        String projectId = parent.getProjectId();
        String parentId = parent.getId();
        String alertId = builder.id == null
                ? MultiAlertUtils.getAlertId(parentId, groupKey)
                : builder.id;
        this.key = AlertKeyInterner.I.intern(new AlertKey(projectId, parentId, alertId));
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public AlertKey getKey() {
        return key;
    }

    @Override
    public String getId() {
        return key.getAlertId();
    }

    @Override
    public String getProjectId() {
        return parent.getProjectId();
    }

    @Override
    public String getFolderId() {
        return parent.getFolderId();
    }

    public Alert getParent() {
        return parent;
    }

    @Override
    public String getName() {
        return parent.getName();
    }

    @Override
    public String getDescription() {
        return parent.getDescription();
    }

    @Override
    public AlertState getState() {
        return parent.getState();
    }

    @Override
    public String getCreatedBy() {
        return parent.getCreatedBy();
    }

    @Override
    public long getCreatedAt() {
        return parent.getCreatedAt();
    }

    @Override
    public String getUpdatedBy() {
        return parent.getUpdatedBy();
    }

    @Override
    public long getUpdatedAt() {
        return parent.getUpdatedAt();
    }

    @Override
    public int getVersion() {
        return parent.getVersion();
    }

    @Override
    public Map<String, ChannelConfig> getNotificationChannels() {
        return parent.getNotificationChannels();
    }

    @Override
    public Set<String> getEscalations() {
        return parent.getEscalations();
    }

    @Override
    public List<String> getGroupByLabels() {
        return Collections.emptyList();
    }

    @Override
    public Map<String, String> getAnnotations() {
        return parent.getAnnotations();
    }

    @Override
    public Map<String, String> getServiceProviderAnnotations() {
        return parent.getServiceProviderAnnotations();
    }

    @Override
    public Map<String, String> getLabels() {
        return parent.getLabels();
    }

    @Override
    public int getDelaySeconds() {
        return parent.getDelaySeconds();
    }

    @Override
    public ResolvedEmptyPolicy getResolvedEmptyPolicy() {
        return parent.getResolvedEmptyPolicy();
    }

    @Override
    public NoPointsPolicy getNoPointsPolicy() {
        return parent.getNoPointsPolicy();
    }

    @Override
    public boolean equalContent(Alert alert) {
        return equals(alert);
    }

    @Override
    public Duration getPeriod() {
        return parent.getPeriod();
    }

    @Override
    public int getMetricsLimit() {
        return parent.getMetricsLimit();
    }

    @Override
    public AlertType getAlertType() {
        return AlertType.SUB_ALERT;
    }

    public Labels getGroupKey() {
        return groupKey;
    }

    @Override
    public AlertSeverity getSeverity() {
        return parent.getSeverity();
    }

    @Override
    public boolean isObtainedFromTemplate() {
        return parent.isObtainedFromTemplate();
    }

    @Override
    public Builder toBuilder() {
        return new Builder(this);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("parent", parent)
                .add("key", key)
                .add("groupKey", groupKey)
                .toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        SubAlert that = (SubAlert) o;
        return key.equals(that.key);
    }

    @Override
    public int hashCode() {
        return key.hashCode();
    }

    @ParametersAreNonnullByDefault
    public static class Builder extends AbstractAlertBuilder<SubAlert, Builder> {
        @Nullable
        private Alert parent;
        @Nullable
        private Labels groupKey;

        private Builder() {
        }

        private Builder(SubAlert alert) {
            super(alert);
            this.parent = alert.parent;
            this.groupKey = alert.groupKey;
        }

        @Override
        protected Builder self() {
            return this;
        }

        public Builder setParent(Alert parent) {
            this.parent = parent;
            return this;
        }

        public Builder setGroupKey(Labels key) {
            this.groupKey = key;
            return this;
        }

        @Override
        public SubAlert build() {
            return new SubAlert(this);
        }
    }
}
