package ru.yandex.solomon.alert.domain.expression;

import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

import ru.yandex.solomon.alert.domain.AbstractAlert;
import ru.yandex.solomon.alert.domain.AbstractAlertBuilder;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.StringInterner;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ExpressionAlert extends AbstractAlert {
    // TODO: group together program and expression after https://st.yandex-team.ru/SOLOMON-2668 (gordiychuk@)
    private final String program;
    private final String checkExpression;

    protected ExpressionAlert(Builder builder) {
        super(builder);
        this.program = Strings.nullToEmpty(builder.program);
        this.checkExpression = Strings.nullToEmpty(builder.checkExpression);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public AlertType getAlertType() {
        return AlertType.EXPRESSION;
    }

    @Override
    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getProgram() {
        return program;
    }

    @Deprecated
    public String getCheckExpression() {
        return checkExpression;
    }

    public static String combineProgramWithCheck(String program, String check) {
        if (check.isEmpty()) {
            return program;
        }
        return program + "\nalarm_if(" + check + ");";
    }

    public String getCombinedSource() {
        return combineProgramWithCheck(program, checkExpression);
    }

    @Override
    public String toString() {
        return "ExpressionAlert{" +
                "program='" + program + '\'' +
                ", checkExpression='" + checkExpression + '\'' +
                "} : " + super.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        ExpressionAlert that = (ExpressionAlert) o;
        return program.equals(that.program) &&
                checkExpression.equals(that.checkExpression);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), program, checkExpression);
    }

    @ParametersAreNonnullByDefault
    public static class Builder extends AbstractAlertBuilder<ExpressionAlert, Builder> {
        @Nullable
        private String program;
        @Nullable
        private String checkExpression;

        public Builder() {
        }

        public Builder(ExpressionAlert alert) {
            super(alert);
            this.program = alert.getProgram();
            this.checkExpression = alert.getCheckExpression();
        }

        public Builder setProgram(String program) {
            this.program = StringInterner.I.intern(Strings.nullToEmpty(program));
            return self();
        }

        public Builder setCheckExpression(String checkExpression) {
            this.checkExpression = StringInterner.I.intern(Strings.nullToEmpty(checkExpression));
            return self();
        }

        @Override
        protected Builder self() {
            return this;
        }

        @Override
        public ExpressionAlert build() {
            return new ExpressionAlert(this);
        }
    }
}
