package ru.yandex.solomon.alert.domain.template;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.domain.AbstractAlert;
import ru.yandex.solomon.alert.domain.AbstractAlertBuilder;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
public class AlertFromTemplatePersistent extends AbstractAlert {
    private final List<AlertParameter> parameters;
    private final List<AlertParameter> thresholds;
    private final String templateId;
    private final String templateVersionTag;
    private final String serviceProvider;
    private Map<String, String> serviceProviderLabels = Map.of();

    protected AlertFromTemplatePersistent(AlertFromTemplatePersistent.Builder builder) {
        super(builder);
        this.parameters = Nullables.orEmpty(builder.parameters);
        this.thresholds = Nullables.orEmpty(builder.thresholds);
        this.serviceProvider = Nullables.orEmpty(builder.serviceProvider);
        this.templateId = Objects.requireNonNull(builder.templateId, "templateId");
        this.templateVersionTag = Objects.requireNonNull(builder.templateVersionTag, "templateVersionTag");
    }

    public static AlertFromTemplatePersistent.Builder newBuilder() {
        return new AlertFromTemplatePersistent.Builder();
    }

    @Override
    public AlertType getAlertType() {
        return AlertType.FROM_TEMPLATE;
    }

    @Override
    public AlertFromTemplatePersistent.Builder toBuilder() {
        return new AlertFromTemplatePersistent.Builder(this);
    }

    public List<AlertParameter> getParameters() {
        return parameters;
    }

    public List<AlertParameter> getThresholds() {
        return thresholds;
    }

    public String getTemplateId() {
        return templateId;
    }

    public String getTemplateVersionTag() {
        return templateVersionTag;
    }

    public String getServiceProvider() {
        return serviceProvider;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        AlertFromTemplatePersistent that = (AlertFromTemplatePersistent) o;
        return Objects.equals(parameters, that.parameters) &&
                Objects.equals(thresholds, that.thresholds) &&
                Objects.equals(templateId, that.templateId) &&
                Objects.equals(templateVersionTag, that.templateVersionTag) &&
                Objects.equals(serviceProviderLabels, that.serviceProviderLabels) &&
                Objects.equals(serviceProvider, that.serviceProvider);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), parameters, thresholds, templateId, templateVersionTag, serviceProvider, serviceProviderLabels);
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", AlertFromTemplatePersistent.class.getSimpleName() + "[", "]")
                .add("parameters=" + parameters)
                .add("thresholds=" + thresholds)
                .add("templateId='" + templateId + "'")
                .add("templateVersionTag='" + templateVersionTag + "'")
                .add("serviceProvider='" + serviceProvider + "'")
                .add("serviceProviderLabels='" + serviceProviderLabels + "'")
                .toString();
    }

    public void setServiceProviderLabels(Map<String, String> labels) {
        serviceProviderLabels = Nullables.orEmpty(labels);
    }

    public Map<String, String> getServiceProviderLabels() {
        return serviceProviderLabels;
    }

    @ParametersAreNonnullByDefault
    public static class Builder extends AbstractAlertBuilder<AlertFromTemplatePersistent, AlertFromTemplatePersistent.Builder> {
        private List<AlertParameter> parameters = List.of();
        private List<AlertParameter> thresholds = List.of();
        private String templateId;
        private String templateVersionTag;
        private String  serviceProvider;

        public Builder() {
        }

        public Builder(AlertFromTemplatePersistent alert) {
            super(alert);
            this.parameters = alert.getParameters();
            this.thresholds = alert.getThresholds();
            this.templateId = alert.getTemplateId();
            this.templateVersionTag = alert.getTemplateVersionTag();
            this.serviceProvider = alert.getServiceProvider();
        }

        public AlertFromTemplatePersistent.Builder setTemplateId(String templateId) {
            this.templateId = templateId;
            return self();
        }

        public AlertFromTemplatePersistent.Builder setTemplateVersionTag(String templateVersionTag) {
            this.templateVersionTag = templateVersionTag;
            return self();
        }

        public AlertFromTemplatePersistent.Builder setParameters(List<AlertParameter> parameters) {
            this.parameters = new ArrayList<>(parameters);
            return self();
        }

        public AlertFromTemplatePersistent.Builder setThresholds(List<AlertParameter> thresholds) {
            this.thresholds = new ArrayList<>(thresholds);
            return self();
        }

        public AlertFromTemplatePersistent.Builder setServiceProvider(String serviceProvider) {
            this.serviceProvider = serviceProvider;
            return self();
        }

        @Override
        protected AlertFromTemplatePersistent.Builder self() {
            return this;
        }

        @Override
        public AlertFromTemplatePersistent build() {
            return new AlertFromTemplatePersistent(this);
        }
    }
}
