package ru.yandex.solomon.alert.domain.template;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter.ParameterValueType;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@JsonTypeInfo(
        use = JsonTypeInfo.Id.NAME,
        property = "type")
@JsonSubTypes({
        @JsonSubTypes.Type(value = AlertParameter.TextParameterValue.class, name = "TEXT"),
        @JsonSubTypes.Type(value = AlertParameter.TextListParameterValue.class, name = "TEXT_LIST"),
        @JsonSubTypes.Type(value = AlertParameter.LabelListParameterValue.class, name = "LABEL_LIST"),
        @JsonSubTypes.Type(value = AlertParameter.DoubleParameterValue.class, name = "DOUBLE"),
        @JsonSubTypes.Type(value = AlertParameter.IntegerParameterValue.class, name = "INTEGER"),
})
public abstract class AlertParameter extends DefaultObject {

    @JsonProperty
    protected final String name;

    public String getName() {
        return name;
    }

    @JsonCreator
    public AlertParameter(@JsonProperty("name") String name) {
        this.name = Nullables.orEmpty(name);
    }

    public abstract ParameterValueType getType();

    public abstract ru.yandex.solomon.alert.protobuf.AlertParameter toProto();

    @JsonIgnore
    public abstract boolean isValueValid();

    public static class DoubleParameterValue extends AlertParameter {
        @JsonProperty
        private final double value;

        @JsonCreator
        public DoubleParameterValue(
                @JsonProperty("value") Double value,
                @JsonProperty("name") String name)
        {
            super(name);
            this.value = Nullables.orZero(value);
        }

        public double getValue() {
            return value;
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.DOUBLE;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertParameter.newBuilder()
                    .setDoubleParameterValue(ru.yandex.solomon.alert.protobuf.AlertParameter.DoubleParameterValue.newBuilder()
                            .setName(name)
                            .setValue(value)
                            .build())
                    .build();
        }

        @Override
        public boolean isValueValid() {
            return true;
        }
    }

    public static class IntegerParameterValue extends AlertParameter {
        @JsonProperty
        private final int value;

        @JsonCreator
        public IntegerParameterValue(
                @JsonProperty("value") Integer value,
                @JsonProperty("name") String name)
        {
            super(name);
            this.value = Nullables.orZero(value);
        }

        public int getValue() {
            return value;
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.INTEGER;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertParameter.newBuilder()
                    .setIntegerParameterValue(ru.yandex.solomon.alert.protobuf.AlertParameter.IntegerParameterValue.newBuilder()
                            .setName(name)
                            .setValue(value)
                            .build())
                    .build();
        }

        @Override
        public boolean isValueValid() {
            return true;
        }
    }

    public static class TextParameterValue extends AlertParameter {
        @JsonProperty
        private final String value;

        @JsonCreator
        public TextParameterValue(
                @JsonProperty("value") String value,
                @JsonProperty("name") String name)
        {
            super(name);
            this.value = Nullables.orEmpty(value);
        }

        public String getValue() {
            return value;
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.TEXT;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertParameter.newBuilder()
                    .setTextParameterValue(ru.yandex.solomon.alert.protobuf.AlertParameter.TextParameterValue.newBuilder()
                            .setName(name)
                            .setValue(value)
                            .build())
                    .build();
        }

        @Override
        public boolean isValueValid() {
            return !value.isEmpty();
        }
    }

    public static class TextListParameterValue extends AlertParameter {
        @JsonProperty
        private final List<String> value;

        @JsonCreator
        public TextListParameterValue(
                @JsonProperty("value") List<String> value,
                @JsonProperty("name") String name)
        {
            super(name);
            this.value = Nullables.orEmpty(value);
        }

        public List<String> getValue() {
            return value;
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.TEXT_LIST;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertParameter.newBuilder()
                    .setTextListParameterValue(ru.yandex.solomon.alert.protobuf.AlertParameter.TextListParameterValue.newBuilder()
                            .setName(name)
                            .addAllValues(value)
                            .build())
                    .build();
        }

        @Override
        public boolean isValueValid() {
            return !value.isEmpty();
        }
    }

    public static class LabelListParameterValue extends AlertParameter {
        @JsonProperty
        private final List<String> values;

        @JsonCreator
        public LabelListParameterValue(
                @JsonProperty("values") List<String> values,
                @JsonProperty("name") String name)
        {
            super(name);
            this.values = Nullables.orEmpty(values);
        }

        public List<String> getValues() {
            return values;
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.LABEL_LIST;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertParameter.newBuilder()
                    .setLabelListParameterValue(ru.yandex.solomon.alert.protobuf.AlertParameter.LabelListParameterValue.newBuilder()
                            .setName(name)
                            .addAllValues(values)
                            .build())
                    .build();
        }

        @Override
        public boolean isValueValid() {
            return !values.isEmpty();
        }
    }
}

