package ru.yandex.solomon.alert.domain.threshold;

import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class PredicateRule {
    private final ThresholdType thresholdType;
    private final double threshold;
    private final Compare comparison;
    private final TargetStatus status;

    public ThresholdType getThresholdType() {
        return thresholdType;
    }

    public double getThreshold() {
        return threshold;
    }

    public Compare getComparison() {
        return comparison;
    }

    public TargetStatus getTargetStatus() {
        return status;
    }

    private PredicateRule(@Nullable ThresholdType thresholdType, double threshold, @Nullable Compare comparison, @Nullable TargetStatus status) {
        this.thresholdType = Nullables.orDefault(thresholdType, ThresholdType.AT_LEAST_ONE);
        this.threshold = threshold;
        this.comparison = Nullables.orDefault(comparison, Compare.GTE);
        this.status = Nullables.orDefault(status, TargetStatus.ALARM);
    }

    public static PredicateRule of(ThresholdType thresholdType, Compare comparison, double threshold, TargetStatus status) {
        return new PredicateRule(thresholdType, threshold, comparison, status);
    }

    public static PredicateRule onThreshold(double threshold) {
        return new PredicateRule(null, threshold, null, null);
    }

    public PredicateRule withThreshold(double threshold) {
        return new PredicateRule(thresholdType, threshold, comparison, status);
    }

    public PredicateRule withThresholdType(ThresholdType thresholdType) {
        return new PredicateRule(thresholdType, threshold, comparison, status);
    }

    public PredicateRule withComparison(Compare comparison) {
        return new PredicateRule(thresholdType, threshold, comparison, status);
    }

    public PredicateRule withTargetStatus(TargetStatus status) {
        return new PredicateRule(thresholdType, threshold, comparison, status);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PredicateRule that = (PredicateRule) o;
        return Double.compare(that.threshold, threshold) == 0 &&
                thresholdType == that.thresholdType &&
                comparison == that.comparison &&
                status == that.status;
    }

    @Override
    public int hashCode() {
        return Objects.hash(thresholdType, threshold, comparison, status);
    }

    @Override
    public String toString() {
        return "PredicateRule{" +
                "thresholdType=" + thresholdType +
                ", threshold=" + threshold +
                ", comparison=" + comparison +
                ", status=" + status +
                '}';
    }

    public static PredicateRule fromProto(TPredicateRule proto) {
        return new PredicateRule(
                ThresholdType.forNumber(proto.getThresholdTypeValue()),
                proto.getThreshold(),
                Compare.forNumber(proto.getComparisonValue()),
                TargetStatus.forNumber(proto.getTargetStatusValue())
            );
    }

    public static TPredicateRule toProto(PredicateRule predicateRule) {
        return TPredicateRule.newBuilder()
                .setComparison(ECompare.forNumber(predicateRule.getComparison().getNumber()))
                .setThresholdType(EThresholdType.forNumber(predicateRule.getThresholdType().getNumber()))
                .setThreshold(predicateRule.getThreshold())
                .setTargetStatus(TPredicateRule.ETargetStatus.forNumber(predicateRule.getTargetStatus().getNumber()))
                .build();
    }
}
