package ru.yandex.solomon.alert.domain.threshold;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

import ru.yandex.solomon.alert.domain.AbstractAlert;
import ru.yandex.solomon.alert.domain.AbstractAlertBuilder;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.StringInterner;
import ru.yandex.solomon.labels.query.Selectors;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ThresholdAlert extends AbstractAlert {
    private final Selectors selectors;
    private final String transformations;
    private final List<PredicateRule> predicateRules;

    protected ThresholdAlert(Builder builder) {
        super(builder);
        this.selectors = requireNonNull(builder.selectors, "Label selector can't null");
        this.transformations = builder.transformations;
        this.predicateRules = builder.predicateRules;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public AlertType getAlertType() {
        return AlertType.THRESHOLD;
    }

    public Selectors getSelectors() {
        return selectors;
    }

    public List<PredicateRule> getPredicateRules() {
        return predicateRules;
    }

    @Deprecated
    public PredicateRule getPredicateRule() {
        return predicateRules.get(0);
    }

    public String getTransformations() {
        return transformations;
    }

    @Override
    public Builder toBuilder() {
        return new Builder(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        ThresholdAlert that = (ThresholdAlert) o;
        return selectors.equals(that.selectors) &&
                transformations.equals(that.transformations) &&
                predicateRules.equals(that.predicateRules);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), selectors, transformations, predicateRules);
    }

    @Override
    public String toString() {
        return "ThresholdAlert{" +
                "selectors=" + selectors +
                ", transformations=" + transformations +
                ", predicateRules=" + predicateRules +
                "} : " + super.toString();
    }

    @ParametersAreNonnullByDefault
    public static class Builder extends AbstractAlertBuilder<ThresholdAlert, Builder> {
        @Nullable
        private Selectors selectors;

        private String transformations = "";
        private List<PredicateRule> predicateRules = new ArrayList<>();

        public Builder() {
        }

        public Builder(ThresholdAlert alert) {
            super(alert);
            this.selectors = alert.getSelectors();
            this.predicateRules = alert.getPredicateRules();
            this.transformations = alert.getTransformations();
        }

        @Override
        protected Builder self() {
            return this;
        }

        public Builder setSelectors(Selectors selectors) {
            this.selectors = selectors;
            return self();
        }

        /**
         * @see Selectors#parse(String)
         */
        public Builder setSelectors(String selectors) {
            return setSelectors(Selectors.parse(selectors));
        }

        public Builder setPredicateRules(Stream<PredicateRule> predicateRules) {
            this.predicateRules = predicateRules.collect(Collectors.toList());
            return self();
        }

        public Builder setPredicateRule(PredicateRule predicateRule) {
            return setPredicateRules(Stream.of(predicateRule));
        }

        public Builder setTransformations(String transformations) {
            this.transformations = StringInterner.I.intern(Strings.nullToEmpty(transformations));
            return self();
        }

        @Override
        public ThresholdAlert build() {
            return new ThresholdAlert(this);
        }

    }

}
