package ru.yandex.solomon.alert.evaluation;

import java.time.Clock;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.dao.ProjectEvaluationLogsDao;
import ru.yandex.solomon.alert.dao.ProjectEvaluationLogsDao.FilterOpts;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

/**
 * @author Vladimir Gordiychuk
 */
public class EvaluationLogsServiceImpl implements EvaluationLogsService {
    private static final long KEEP_LOGS_MILLIS = TimeUnit.DAYS.toMillis(7);
    private static final Logger logger = LoggerFactory.getLogger(EvaluationLogsServiceImpl.class);

    private final String projectId;
    private final ProjectEvaluationLogsDao dao;
    private final EvaluationLogsServiceMetrics metrics;
    private final Clock clock;
    private final ActorWithFutureRunner actor;
    private final ArrayListLockQueue<EvaluationState> queue = new ArrayListLockQueue<>();

    private volatile boolean closed;
    private volatile long deleteFrom;
    private volatile CompletableFuture<Void> actSync = new CompletableFuture<>();

    public EvaluationLogsServiceImpl(
            String projectId,
            ProjectEvaluationLogsDao dao,
            MetricRegistry registry,
            ExecutorService executorService,
            Clock clock)
    {
        this.projectId = projectId;
        this.dao = dao;
        this.metrics = new EvaluationLogsServiceMetrics(registry);
        this.clock = clock;
        this.actor = new ActorWithFutureRunner(this::act, executorService);
    }

    @Override
    public void save(EvaluationState state) {
        metrics.queueSize.add(1L);
        queue.enqueue(state);
        actor.schedule();
    }

    @Override
    public CompletableFuture<Void> restore(long fromTsMillis, Consumer<EvaluationState> consumer) {
        return dao.findMany(new FilterOpts()
                .setFromMillis(fromTsMillis)
                .setToMillis(clock.millis()), consumer);
    }

    @Override
    public CompletableFuture<Void> actSync() {
        actSync = new CompletableFuture<>();
        actor.schedule();
        return actSync;
    }

    @Override
    public void close() {
        closed = true;
        actor.schedule();
    }

    private CompletableFuture<Void> act() {
        CompletableFuture<Void> syncCopy = actSync;
        return actPersistLogs()
                .thenCompose(ignore -> actDeleteOldLogs())
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        syncCopy.completeExceptionally(e);
                    } else {
                        syncCopy.complete(null);
                    }
                });
    }

    private CompletableFuture<Void> actPersistLogs() {
        List<EvaluationState> batch = queue.dequeueAll();
        metrics.queueSize.add(-batch.size());
        if (closed) {
            return CompletableFuture.completedFuture(null);
        }
        int batchSize = batch.size();
        CompletableFuture<Void> future = dao.save(batch);
        metrics.save.forFuture(future, batchSize);
        return future.handle((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed persist {} logs for project {}", batchSize, projectId, e);
                    } else {
                        logger.debug("Saved {} logs for project {}", batchSize, projectId);
                    }

                    return null;
                });
    }

    private CompletableFuture<Void> actDeleteOldLogs() {
        long now = roundToMin(clock.millis());
        long deleteTo = now - KEEP_LOGS_MILLIS;
        if (deleteFrom == deleteTo || closed) {
            return CompletableFuture.completedFuture(null);
        }

        CompletableFuture<Void> future = dao.delete(new FilterOpts()
                .setFromMillis(deleteFrom)
                .setToMillis(deleteTo));
        metrics.delete.forFuture(future);
        return future.handle((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed delete old logs for project {}", projectId, e);
                    } else {
                        deleteFrom = deleteTo;
                    }

                    return null;
                });
    }

    private long roundToMin(long tsMillis) {
        return tsMillis - (tsMillis % 60_000);
    }
}
