package ru.yandex.solomon.alert.evaluation;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.api.AlertingException;
import ru.yandex.solomon.alert.api.ErrorClassificatory;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.cluster.AlertingShardProxy;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.TemplateAlertFactory;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlertTimeSeries;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.alert.rule.AlertRule;
import ru.yandex.solomon.alert.rule.AlertRuleDeadlines;
import ru.yandex.solomon.alert.rule.AlertRuleDefaultDeadlines;
import ru.yandex.solomon.alert.rule.AlertRuleFactory;
import ru.yandex.solomon.alert.rule.EmptyTemplateProcessor;
import ru.yandex.solomon.alert.rule.ExplainResult;
import ru.yandex.solomon.alert.rule.SimulationResult;
import ru.yandex.solomon.alert.rule.SimulationStatus;
import ru.yandex.solomon.alert.unroll.MultiAlertUnrollFactory;
import ru.yandex.solomon.alert.unroll.MultiAlertUtils;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.GraphData;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class LocalEvaluationExplainService implements EvaluationExplainService {
    private final AlertingShardProxy proxy;
    private final AlertRuleFactory ruleFactory;
    private final MultiAlertUnrollFactory unrollFactory;
    private final AlertTemplateDao alertTemplateDao;
    private final TemplateAlertFactory templateAlertFactory;

    public LocalEvaluationExplainService(
            AlertingShardProxy proxy,
            AlertRuleFactory ruleFactory,
            MultiAlertUnrollFactory unrollFactory,
            AlertTemplateDao cachedAlertTemplateDao,
            TemplateAlertFactory templateAlertFactory)
    {
        this.proxy = proxy;
        this.ruleFactory = ruleFactory;
        this.unrollFactory = unrollFactory;
        this.alertTemplateDao = cachedAlertTemplateDao;
        this.templateAlertFactory = templateAlertFactory;
    }

    @Override
    public CompletableFuture<TExplainEvaluationResponse> explainEvaluation(TExplainEvaluationRequest request) {
        return CompletableFutures.safeCall(() -> getAlert(request))
                .thenCompose(this::prepareAlert)
                .thenCompose(alert -> {
                    Instant now = Instant.ofEpochMilli(request.getEvaluationTimeMillis());
                    Instant deadline = request.getDeadlineMillis() == 0
                            ? Instant.now().plusSeconds(30L)
                            : Instant.ofEpochMilli(request.getDeadlineMillis());

                    return explain(alert, now, deadline);
                })
                .thenApply(explain -> {
                    EvaluationStatus status = explain.getStatus();
                    List<TAlertTimeSeries> series = explain.getSeries()
                            .stream()
                            .filter(ts -> !isPrefetchAlias(ts.getAlias()))
                            .map(ts -> {
                                GraphData graphData = AggrGraphDataArrayList.of(ts.getSource().iterator())
                                        .toGraphDataShort(ts.getDataType());

                                return TAlertTimeSeries.newBuilder()
                                        .addAllLabels(LabelConverter.labelsToProtoList(ts.getLabels()))
                                        .setAlias(ts.getAlias())
                                        .addAllTimeMillis(graphData.getTimestamps()
                                                .stream()
                                                .boxed()
                                                .collect(Collectors.toList()))
                                        .addAllValues(graphData.getValues()
                                                .stream()
                                                .boxed()
                                                .collect(Collectors.toList()))
                                        .build();
                            }).collect(Collectors.toList());

                    return TExplainEvaluationResponse.newBuilder()
                            .setRequestStatus(ERequestStatusCode.OK)
                            .setEvaluationStatus(AlertConverter.statusToProto(status))
                            .addAllSeries(series)
                            .build();
                })
                .exceptionally(e -> {
                    ERequestStatusCode code = ErrorClassificatory.classifyError(e);
                    return TExplainEvaluationResponse.newBuilder()
                            .setRequestStatus(code)
                            .setStatusMessage(Throwables.getStackTraceAsString(e))
                            .build();
                });
    }

    private CompletableFuture<Alert> prepareAlert(Alert alert) {
        if (alert.getAlertType() != AlertType.FROM_TEMPLATE) {
            return CompletableFuture.completedFuture(alert);
        }
        AlertFromTemplatePersistent alertFromTemplate = (AlertFromTemplatePersistent) alert;
        return alertTemplateDao.findById(alertFromTemplate.getTemplateId(), alertFromTemplate.getTemplateVersionTag())
                .thenApply(alertTemplate -> {
                    if (alertTemplate.isEmpty()) {
                        throw new IllegalArgumentException(
                                String.format("Hasn't alert template with %s %s", alertFromTemplate.getTemplateId(), alertFromTemplate.getTemplateVersionTag()));
                    }
                    return templateAlertFactory.createAlertFromTemplate(alertFromTemplate, alertTemplate.get());
                });
    }

    @Override
    public CompletableFuture<TSimulateEvaluationResponse> simulateEvaluation(TSimulateEvaluationRequest request) {
        return CompletableFutures.safeCall(() -> getAlert(request))
            .thenCompose(this::prepareAlert)
            .thenCompose(alert -> {
                Instant deadline = request.getDeadlineMillis() == 0
                    ? Instant.now().plusSeconds(30L)
                    : Instant.ofEpochMilli(request.getDeadlineMillis());

                long gridMillis = request.getGridMillis();
                Duration gridStep = Duration.ofMillis(gridMillis);
                Instant from = Instant.ofEpochMilli(request.getSimulationTimeBeginMillis());
                Instant to = Instant.ofEpochMilli(request.getSimulationTimeEndMillis());

                return simulate(alert, from, to, gridStep, deadline);
            })
            .thenApply(SimulationResult::toProto);
    }

    // Prefetch variables are internal entities and shouldn't be shown outside
    private static boolean isPrefetchAlias(String alias) {
        return alias.startsWith("prefetch$");
    }

    private CompletableFuture<ExplainResult> explain(Alert alertOrMultiAlert, Instant now, Instant deadline) {
        CompletableFuture<Alert> explainTarget = MultiAlertUnrollFactory.isSupportUnrolling(alertOrMultiAlert) ?
            unroll(alertOrMultiAlert, deadline) : completedFuture(alertOrMultiAlert);

        return explainTarget.thenCompose(alert -> {
            if (alert == null) {
                EvaluationStatus status = EvaluationStatus.NO_DATA
                        .withDescription("Not found sub alerts by grouping labels: " + alertOrMultiAlert.getGroupByLabels());
                return completedFuture(EmptyTemplateProcessor.I.processTemplate(now, status));
            }
            AlertRule rule = ruleFactory.createAlertRule(alert);
            return rule.explain(now, AlertRuleDefaultDeadlines.of(deadline));
        });
    }

    private CompletableFuture<SimulationResult> simulate(Alert alertOrMultiAlert, Instant from, Instant to, Duration gridStep, Instant deadline) {
        CompletableFuture<Alert> simulateTarget = MultiAlertUnrollFactory.isSupportUnrolling(alertOrMultiAlert) ?
            unroll(alertOrMultiAlert, deadline) : completedFuture(alertOrMultiAlert);

        return simulateTarget.thenCompose(alert -> {
            if (alert == null) {
                SimulationResult result = SimulationStatus.NO_SUBALERTS
                        .withMessage("Not found sub alerts by grouping labels: " + alertOrMultiAlert.getGroupByLabels());
                return completedFuture(result);
            }
            AlertRule rule = ruleFactory.createAlertRule(alert);
            return rule.simulate(from, to, gridStep, AlertRuleDefaultDeadlines.of(deadline));
        });
    }

    private CompletableFuture<Alert> unroll(Alert alert, Instant deadline) {
        AlertRuleDeadlines deadlines = AlertRuleDefaultDeadlines.of(deadline);

        return unrollFactory.create(alert)
                .unroll(deadlines)
                .thenApply(groupKeys -> {
                    if (groupKeys.labels.isEmpty()) {
                        return null;
                    }

                    Labels labels = groupKeys.labels.iterator().next();
                    return SubAlert.newBuilder()
                            .setId(MultiAlertUtils.getAlertId(alert, labels))
                            .setParent(alert)
                            .setGroupKey(labels)
                            .build();
                });
    }


    private CompletableFuture<Alert> getAlert(TExplainEvaluationRequest request) {
        switch (request.getCheckCase()) {
            case ALERT:
                return completedFuture(AlertConverter.protoToAlert(request.getAlert()));
            case SUBALERT:
                return completedFuture(AlertConverter.protoToSubAlert(request.getSubAlert()));
            case REFERENCE: {
                return loadAlertByReference(request.getReference(), request.getDeadlineMillis());
            }
            default: {
                return failedFuture(new AlertingException(ERequestStatusCode.INVALID_REQUEST, "Unsupported check cause: " + request));
            }
        }
    }

    private CompletableFuture<Alert> getAlert(TSimulateEvaluationRequest request) {
        switch (request.getCheckCase()) {
            case ALERT:
                return completedFuture(AlertConverter.protoToAlert(request.getAlert()));
            case SUBALERT:
                return completedFuture(AlertConverter.protoToSubAlert(request.getSubAlert()));
            case REFERENCE: {
                return loadAlertByReference(request.getReference(), request.getDeadlineMillis());
            }
            default: {
                return failedFuture(new AlertingException(ERequestStatusCode.INVALID_REQUEST, "Unsupported check cause: " + request));
            }
        }
    }

    private CompletableFuture<Alert> loadAlertByReference(TExplainEvaluationRequest.TReferenceToAlert reference, long deadlineMillis) {
        if (StringUtils.isEmpty(reference.getParentId())) {
            TReadAlertRequest readRequest = TReadAlertRequest.newBuilder()
                    .setProjectId(reference.getProjectId())
                    .setFolderId(reference.getFolderId())
                    .setAlertId(reference.getAlertId())
                    .setDeadlineMillis(deadlineMillis)
                    .build();

            return loadAlert(readRequest);
        } else {
            TReadSubAlertRequest readRequest = TReadSubAlertRequest.newBuilder()
                    .setProjectId(reference.getProjectId())
                    .setFolderId(reference.getFolderId())
                    .setAlertId(reference.getAlertId())
                    .setParentId(reference.getParentId())
                    .setDeadlineMillis(deadlineMillis)
                    .build();

            return loadSubAlert(readRequest);
        }
    }

    private CompletableFuture<Alert> loadAlert(TReadAlertRequest readRequest) {
        return proxy.readAlert(readRequest)
                .thenApply(response -> {
                    if (response.getRequestStatus() != ERequestStatusCode.OK) {
                        throw new AlertingException(response.getRequestStatus(), response.getStatusMessage());
                    }

                    return AlertConverter.protoToAlert(response.getAlert());
                });
    }

    private CompletableFuture<Alert> loadSubAlert(TReadSubAlertRequest request) {
        return proxy.readSubAlert(request)
                .thenApply(response -> {
                    if (response.getRequestStatus() != ERequestStatusCode.OK) {
                        throw new AlertingException(response.getRequestStatus(), response.getStatusMessage());
                    }

                    return AlertConverter.protoToSubAlert(response.getSubAlert());
                });
    }
}
