package ru.yandex.solomon.alert.evaluation;

import java.time.Instant;

import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.rule.AlertRule;
import ru.yandex.solomon.alert.rule.EvaluationState;

/**
 * @author Vladimir Gordiychuk
 */
@ThreadSafe
final class Task implements Comparable<Task> {
    private final int attempt;
    private final long evaluateAt;
    private final long iterationTimeMillis;
    private final AlertRule rule;
    @Nullable
    private final EvaluationState state;
    private final EvaluationService.Consumer consumer;
    private volatile boolean active = true;

    Task(long iterationTimeMillis, AlertRule rule, @Nullable EvaluationState state, EvaluationService.Consumer consumer) {
        this(iterationTimeMillis, iterationTimeMillis, rule, state, consumer);
    }

    Task(long evaluateAt, long iterationTimeMillis, AlertRule rule, @Nullable EvaluationState state, EvaluationService.Consumer consumer) {
        this(evaluateAt, 0, iterationTimeMillis, rule, state, consumer);
    }

    private Task(long evaluateAt, int attempt, long iterationTimeMillis, AlertRule rule, @Nullable EvaluationState state, EvaluationService.Consumer consumer) {
        this.evaluateAt = evaluateAt;
        this.attempt = attempt;
        this.iterationTimeMillis = iterationTimeMillis;
        this.rule = rule;
        this.state = state;
        this.consumer = consumer;
    }

    AlertKey getAlertKey() {
        return rule.getAlert().getKey();
    }

    AlertRule getRule() {
        return rule;
    }

    Alert getAlert() {
        return rule.getAlert();
    }

    @Nullable
    EvaluationState getState() {
        return state;
    }

    @Nullable
    EvaluationStatus getStatus() {
        if (state == null) {
            return null;
        }

        return state.getStatus();
    }

    EvaluationState nextState(EvaluationStatus status) {
        if (state == null) {
            return EvaluationState.newBuilder(rule.getAlert())
                .setStatus(status)
                .setSince(getIterationTime())
                .build();
        } else {
            return state.nextStatus(status, getIterationTime());
        }
    }

    public EvaluationService.Consumer getConsumer() {
        return consumer;
    }

    boolean isActive() {
        return active && !consumer.isCanceled();
    }

    void cancel() {
        active = false;
        consumer.onComplete();
    }

    Task withEvalTime(long evaluateAt, long iterationTimeMillis, EvaluationState state) {
        return new Task(evaluateAt, iterationTimeMillis, rule, state, consumer);
    }

    Task withRetry(long evaluateAt, long iterationTimeMillis) {
        return new Task(evaluateAt, attempt + 1, iterationTimeMillis, rule, state, consumer);
    }

    @Override
    public int compareTo(Task o) {
        return Long.compare(evaluateAt, o.evaluateAt);
    }

    public Instant getIterationTime() {
        return Instant.ofEpochMilli(iterationTimeMillis);
    }

    public long getEvaluateAt() {
        return evaluateAt;
    }

    public int getAttempt() {
        return attempt;
    }

    public long getIterationTimeMillis() {
        return iterationTimeMillis;
    }
}
