package ru.yandex.solomon.alert.evaluation;

import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;

/**
 * Project-wide metrics, without executor label
 * @author Vladimir Gordiychuk
 */
final class TaskEvaluationMetrics {
    private static final Set<AlertType> SUPPORTED_ALERT_TYPES = EnumSet.of(AlertType.THRESHOLD, AlertType.EXPRESSION);

    private final MetricRegistry registry;
    private final ConcurrentMap<String, ProjectMetrics> metricsByProject = new ConcurrentHashMap<>();
    private final ProjectMetrics total;

    TaskEvaluationMetrics(MetricRegistry registry) {
        this.registry = registry;
        this.total = new ProjectMetrics("total", registry);
    }

    void startEval(Alert alert, long iterationMillis) {
        long lagMillis = System.currentTimeMillis() - iterationMillis;
        getProjectMetrics(alert).startEval(alert, lagMillis);
        total.startEval(alert, lagMillis);
    }

    void completeEval(Alert alert, @Nullable EvaluationStatus prevStatus, EvaluationStatus status, long startTimeNano) {
        long elapsedTime = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNano);
        getProjectMetrics(alert).completeEval(alert, prevStatus, status, elapsedTime);
        total.completeEval(alert, prevStatus, status, elapsedTime);
    }

    public void evalTimeout(Alert alert) {
        getProjectMetrics(alert).evalTimeout(alert);
        total.evalTimeout(alert);
    }

    private ProjectMetrics getProjectMetrics(Alert alert) {
        return metricsByProject.computeIfAbsent(alert.getProjectId(), id -> new ProjectMetrics(id, registry));
    }

    private static class ProjectMetrics {
        private final EnumMap<AlertType, TypedMetric> metricsByType;
        private final TypedMetric total;

        public ProjectMetrics(String projectId, MetricRegistry registry) {
            this(registry.subRegistry("projectId", projectId));
        }

        private ProjectMetrics(MetricRegistry registry) {
            EnumMap<AlertType, TypedMetric> metrics = new EnumMap<>(AlertType.class);
            for (AlertType type : SUPPORTED_ALERT_TYPES) {
                metrics.put(type, new TypedMetric(type, registry));
            }
            this.metricsByType = metrics;
            this.total = new TypedMetric(registry.subRegistry("alertType", "total"));
        }

        void startEval(Alert alert, long lagMillis) {
            getAlertTypeMetrics(alert).startEval(lagMillis);
            total.startEval(lagMillis);
        }

        void completeEval(Alert alert, EvaluationStatus prevStatus, EvaluationStatus status, long elapsedTime) {
            getAlertTypeMetrics(alert).completeEval(prevStatus, status, elapsedTime);
            total.completeEval(prevStatus, status, elapsedTime);
        }

        public void evalTimeout(Alert alert) {
            getAlertTypeMetrics(alert).evalTimeout();
            total.evalTimeout();
        }

        private TypedMetric getAlertTypeMetrics(Alert alert) {
            return metricsByType.get(getEffectiveType(alert));
        }

        private AlertType getEffectiveType(Alert alert) {
            AlertType type = alert.getAlertType();
            if (type != AlertType.SUB_ALERT) {
                return type;
            }

            SubAlert subAlert = (SubAlert) alert;
            return subAlert.getParent().getAlertType();
        }
    }

    private static class TypedMetric {
        private final Rate started;
        private final Rate completed;
        private final Rate timeout;
        private final Histogram evalTime;
        private final Histogram evalLag;
        private final EnumMap<EvaluationStatus.Code, Rate> countEvalStatus;
        private final EnumMap<EvaluationStatus.Code, Rate> statusChange;
        private final Rate statusChangeTotal;

        public TypedMetric(AlertType type, MetricRegistry registry) {
            this(registry.subRegistry(Labels.of("alertType", type.name())));
        }

        public TypedMetric(MetricRegistry registry) {
            this.started = registry.rate("evaluations.eval.started");
            this.completed = registry.rate("evaluations.eval.completed");
            this.timeout = registry.rate("evaluations.eval.timeout");
            registry.lazyGaugeInt64("evaluations.eval.inFlight", () -> started.get() - completed.get());
            this.evalTime = registry.histogramRate("evaluations.eval.elapsedTimeMillis",
                    Histograms.exponential(18, 2, 1));
            this.evalLag = registry.histogramRate("evaluations.eval.lagSeconds",
                    Histograms.exponential(12, 2, 1));

            EnumMap<EvaluationStatus.Code, Rate> statuses = new EnumMap<>(EvaluationStatus.Code.class);
            EnumMap<EvaluationStatus.Code, Rate> changes = new EnumMap<>(EvaluationStatus.Code.class);
            for (EvaluationStatus.Code code : EvaluationStatus.Code.values()) {
                Labels labels = Labels.of("status", code.name());
                statuses.put(code, registry.rate("evaluations.eval.status.total", labels));
                changes.put(code, registry.rate("evaluations.eval.status.change", Labels.of("target_status", code.name())));
            }
            countEvalStatus = statuses;
            statusChange = changes;
            statusChangeTotal = registry.rate("evaluations.eval.status.change", Labels.of("target_status", "total"));
        }

        void startEval(long lagMillis) {
            started.inc();
            evalLag.record(TimeUnit.MILLISECONDS.toSeconds(lagMillis));
        }

        void completeEval(@Nullable EvaluationStatus prevStatus, EvaluationStatus status, long elapsedTime) {
            evalTime.record(elapsedTime);
            completed.inc();
            countEvalStatus.get(status.getCode()).inc();
            if (prevStatus == null || prevStatus.getCode() != status.getCode()) {
                statusChange.get(status.getCode()).inc();
                statusChangeTotal.inc();
            }
        }

        void evalTimeout() {
            timeout.inc();
        }
    }
}
