package ru.yandex.solomon.alert.evaluation;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.context.annotation.Bean;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
public class TaskExecutorByProjectWww {
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final ManagerWriterContext context;
    private final TaskExecutorDispatcher executor;

    public TaskExecutorByProjectWww(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            ManagerWriterContext context,
            TaskExecutorDispatcher executor)
    {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.context = context;
        this.executor = executor;
    }

    @Bean
    public RootLink evaluationTaskByProject() {
        return new RootLink("/alerting-local-evaluation-by-project", "Alerting local evaluation by project");
    }

    @RequestMapping(value = "/alerting-local-evaluation-by-project", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localEvaluationProjectSummary(
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> localEvaluationProjectSummaryImpl(sortBy));
    }

    private String localEvaluationProjectSummaryImpl(int sortBy) {
        List<Record> records = this.executor.getTasksStream()
            .filter(entry -> entry.getValue().isActive())
            .map(Record::of)
            .collect(Collectors.groupingBy(r -> r.projectId))
            .values()
            .stream()
            .map(recordList -> recordList.stream().reduce(new Record(), Record::combine))
            .sorted(Comparator.comparing(r -> r.projectId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Local evaluation by projects", context, columns, records, sortBy).genString();
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
            Column.of(
                "ProjectId",
                r -> new AHref("/alerting-local-evaluation/projects/" + r.projectId, r.projectId),
                Comparator.comparing(o -> o.projectId)),
            Column.of("Alerts", r -> r.alerts, Comparator.comparing(o2 -> o2.alerts)),
            Column.of("Attempt", r -> r.attempt, Comparator.comparing(o2 -> o2.attempt)),
            Column.of("EvaluationLag",
                r -> DurationUtils.formatDurationMillis(r.evaluationLag),
                Comparator.comparingLong(o2 -> o2.evaluationLag)),
            Column.of("MaxEvaluationLag",
                r -> DurationUtils.formatDurationMillis(r.maxEvaluationLag),
                Comparator.comparingLong(o2 -> o2.maxEvaluationLag)),
            Column.of("IoTime/min",
                r -> DurationUtils.formatDurationNanos(r.ioTimeNanos),
                Comparator.comparingLong(o2 -> o2.ioTimeNanos)),
            Column.of("CpuTime/min",
                r -> DurationUtils.formatDurationNanos(r.cpuTimeNanos),
                Comparator.comparingLong(o2 -> o2.cpuTimeNanos))
        );
    }

    public static class Record implements TableRecord {
        String projectId;
        int alerts;
        int attempt;
        long evaluationLag;
        long maxEvaluationLag;
        Meter ioTime = Meter.of(ExpMovingAverage.fifteenMinutes());
        Meter cpuTime = Meter.of(ExpMovingAverage.fifteenMinutes());
        long cpuTimeNanos;
        long ioTimeNanos;

        public static Record of(Map.Entry<AlertKey, Task> entry) {
            var r = new Record();
            r.projectId = entry.getKey().getProjectId();
            r.alerts = 1;
            r.attempt = entry.getValue().getAttempt();
            r.evaluationLag =  System.currentTimeMillis() - entry.getValue().getEvaluateAt();
            if (r.evaluationLag < 0) {
                r.evaluationLag = 0;
            }
            r.maxEvaluationLag = r.evaluationLag;
            r.ioTime = entry.getValue().getRule().getIoTime();
            r.cpuTime = entry.getValue().getRule().getCpuTime();
            return r;
        }

        public Record combine(Record record) {
            this.projectId = record.projectId;
            this.attempt += record.attempt;
            this.alerts += record.alerts;
            this.evaluationLag += record.evaluationLag;
            this.maxEvaluationLag = Math.max(this.maxEvaluationLag, record.maxEvaluationLag);
            this.ioTime.combine(record.ioTime);
            this.cpuTime.combine(record.cpuTime);
            this.ioTimeNanos += nanoToMillis(ioTime);
            this.cpuTimeNanos += nanoToMillis(cpuTime);
            return this;
        }

        private static long nanoToMillis(Meter meter) {
            return Math.round(meter.getRate(TimeUnit.MINUTES));
        }
    }
}
