package ru.yandex.solomon.alert.evaluation;

import java.time.Clock;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.WillNotClose;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.rule.AlertRule;
import ru.yandex.solomon.alert.rule.EvaluationState;

public class TaskExecutorDispatcher {
    private final EnumMap<AlertRuleExecutorType, TaskExecutorImpl> executors;

    private final Clock clock;
    @WillNotClose
    private final ExecutorService executor; // Actual executor is the same for all TaskExecutors. This may change in future
    @WillNotClose
    private final ScheduledExecutorService timer;
    private final AlertExecutorOptionsProvider optionsFactory;
    private final MetricRegistry registry;
    private final TaskEvaluationMetrics evaluationMetrics;

    public TaskExecutorDispatcher(
            Clock clock,
            @WillNotClose ExecutorService executor,
            @WillNotClose ScheduledExecutorService timer,
            AlertExecutorOptionsProvider optionsFactory,
            MetricRegistry registry)
    {
        this.clock = clock;
        this.executor = executor;
        this.timer = timer;
        this.optionsFactory = optionsFactory;
        this.registry = registry;
        this.evaluationMetrics = new TaskEvaluationMetrics(registry);

        executors = new EnumMap<>(Arrays.stream(AlertRuleExecutorType.values())
                .collect(Collectors.toUnmodifiableMap(
                        Function.identity(),
                        this::makeExecutorForType
                )));
    }

    public double getTotalEvaluationRate() {
        return executors.values().stream()
                .map(TaskExecutorImpl::getMetrics)
                .mapToDouble(TaskExecutorMetrics::getEvaluationRate)
                .sum();
    }

    public int getTaskCount() {
        return executors.values().stream()
                .map(TaskExecutorImpl::getTasks)
                .mapToInt(Map::size)
                .sum();
    }

    private TaskExecutorImpl makeExecutorForType(AlertRuleExecutorType type) {
        var options = optionsFactory.getOptionsFor(type);
        return new TaskExecutorImpl(
                clock,
                executor,
                timer,
                options,
                registry.subRegistry("executor", type.name()),
                evaluationMetrics);
    }

    @VisibleForTesting
    public void scheduleAct() {
        executors.values().forEach(TaskExecutorImpl::scheduleAct);
    }

    public void createAndScheduleTask(AlertRule rule, EvaluationState state, EvaluationService.Consumer consumer) {
        if (state != null && state.getAlertVersion() != rule.getAlert().getVersion()) {
            // state already obsoleted
            state = null;
        }

        var type = rule.getExecutorType();

        var random = ThreadLocalRandom.current();
        final long iterationTime;
        final long evaluationTime;
        final long now = clock.millis();
        var options = optionsFactory.getOptionsFor(type);
        if (state == null) {
            iterationTime = now + random.nextLong(0, options.getEvalIntervalMillis());
            evaluationTime = iterationTime;
        } else {
            iterationTime = state.getLatestEval().toEpochMilli() + options.getEvalIntervalMillis();
            evaluationTime = Math.max(now, iterationTime) + random.nextLong(options.getMaxEvaluationJitterMillis());
        }

        Task task = new Task(evaluationTime, iterationTime, rule, state, consumer);
        executors.get(type).scheduleNewTask(task);
    }

    public boolean cancelTask(AlertKey alertKey) {
        return executors.values().stream().anyMatch(executor -> executor.cancelTask(alertKey));
    }

    public void close() {
        executors.values().forEach(TaskExecutorImpl::close);
    }

    public Stream<Map.Entry<AlertKey, Task>> getTasksStream() {
        return executors.values().stream()
                .flatMap(ex -> ex.getTasks().entrySet().stream());
    }

    @Nullable
    public Task getTask(AlertKey key) {
        return executors.values().stream()
                .flatMap(ex -> Optional.ofNullable(ex.getTasks().get(key)).stream())
                .findFirst()
                .orElse(null);
    }
}
