package ru.yandex.solomon.alert.evaluation;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import org.springframework.context.annotation.Bean;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.rule.AlertRule;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.find.annotation.NamedObjectFinderAnnotation;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
public class TaskExecutorWww {
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final ManagerWriterContext context;
    private final TaskExecutorDispatcher executor;

    public TaskExecutorWww(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            ManagerWriterContext context,
            TaskExecutorDispatcher executor)
    {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.context = context;
        this.executor = executor;
    }

    @Bean
    public RootLink evaluationTask() {
        return new RootLink("/alerting-local-evaluation", "Alerting local evaluation");
    }

    @RequestMapping(value = "/alerting-local-evaluation", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localEvaluation(
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> localEvaluationImpl(sortBy));
    }

    private String localEvaluationImpl(int sortBy) {
        List<Record> records = this.executor.getTasksStream()
            .filter(entry -> entry.getValue().isActive())
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.alertId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Local evaluation", context, columns, records, sortBy, 1000).genString();
    }

    @RequestMapping(value = "/alerting-local-evaluation/projects/{projectId}/alerts/{parentId}", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localEvaluationByParent(
        @PathVariable("projectId") String projectId,
        @PathVariable("parentId") String parentId,
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> localEvaluationByParentImpl(projectId, parentId, sortBy));
    }

    private String localEvaluationByParentImpl(String projectId, String parentId, int sortBy) {
        List<Record> records = this.executor.getTasksStream()
            .filter(entry -> entry.getValue().isActive())
            .filter(entry -> {
                var key = entry.getKey();
                return key.getProjectId().equals(projectId) && key.getParentId().equals(parentStrToId(parentId));
            })
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.alertId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Local evaluation by " + projectId + "/" + parentId, context, columns, records, sortBy, 1000).genString();
    }

    @RequestMapping(value = "/alerting-local-evaluation/projects/{projectId}/alerts/{parentId}/{alertId}", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localEvaluationByAlertId(
        @PathVariable("projectId") String projectId,
        @PathVariable("parentId") String parentId,
        @PathVariable("alertId") String alertId,
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> localEvaluationByAlertIdImpl(projectId, parentId, alertId, sortBy));
    }

    private String localEvaluationByAlertIdImpl(String projectId, String parentId, String alertId, int sortBy) {
        List<Record> records = this.executor.getTasksStream()
            .filter(entry -> entry.getValue().isActive())
            .filter(entry -> {
                var key = entry.getKey();
                return key.getProjectId().equals(projectId)
                    && key.getParentId().equals(parentStrToId(parentId))
                    && key.getAlertId().equals(alertId);
            })
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.alertId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Local evaluation by " + projectId + "/" + parentId, context, columns, records, sortBy, 1000).genString();
    }

    @RequestMapping(value = "/alerting-local-evaluation/projects/{projectId}", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> localEvaluationByProject(
        @PathVariable("projectId") String projectId,
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> localEvaluatioByProject(projectId, sortBy));
    }

    private String localEvaluatioByProject(String projectId, int sortBy) {
        List<Record> records = this.executor.getTasksStream()
            .filter(entry -> entry.getValue().isActive())
            .filter(entry -> {
                var key = entry.getKey();
                return key.getProjectId().equals(projectId);
            })
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.alertId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Local evaluation by " + projectId, context, columns, records, sortBy, 1000).genString();
    }

    @Nullable
    @NamedObjectFinderAnnotation
    public AlertRule getAlertRule(String id) {
        var task = getTask(id);
        if (task != null) {
            return task.getRule();
        }
        return null;
    }

    @Nullable
    @NamedObjectFinderAnnotation
    public Task getTask(String id) {
        String[] parts = id.split(":");
        String projectId = parts[0];
        String parentId = parts[1];
        String alertId = parts[2];

        AlertKey key = new AlertKey(projectId, parentStrToId(parentId), alertId);
        return executor.getTask(key);
    }

    private String makeStrKey(String projectId, String parentId, String alertId) {
        return projectId + ":" + parentId + ":" + alertId;
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
            Column.of(
                "ProjectId",
                r -> {
                    return new AHref("/alerting-local-evaluation/projects/" + r.projectId, r.projectId);
                },
                Comparator.comparing(o -> o.projectId)),
            Column.of(
                "ParentId",
                r -> {
                    return new AHref("/alerting-local-evaluation/projects/" + r.projectId + "/alerts/" + r.parentId, r.parentId);
                },
                Comparator.comparing(o -> o.parentId)),
            Column.of(
                "SubAlertId",
                r -> {
                    String id = r.alertId;
                    return new AHref("/alerting-local-evaluation/projects/" + r.projectId + "/alerts/" + r.parentId + "/" + id, id);
                },
                Comparator.comparing(o -> o.alertId)),
            Column.of(
                "Task",
                r -> {
                    String key = makeStrKey(r.projectId, r.parentId, r.alertId);
                    return new AHref(namedObjectLink(new NamedObjectId(Task.class, key)), "Task");
                },
                Comparator.comparing(o -> o.alertId)),
            Column.of("Attempt", r -> r.attempt, Comparator.comparing(o2 -> o2.attempt)),
            Column.of("EvaluationLag",
                r -> DurationUtils.formatDurationMillis(r.evaluationLag),
                Comparator.comparingLong(o2 -> o2.evaluationLag)),
            Column.of("IoTime/min",
                r -> DurationUtils.formatDurationNanos(r.ioTime),
                Comparator.comparingLong(o2 -> o2.ioTime)),
            Column.of("CpuTime/min",
                r -> DurationUtils.formatDurationNanos(r.cpuTime),
                Comparator.comparingLong(o2 -> o2.cpuTime)),
            Column.of("Status",
                r -> r.statusCode == null ? "unknown" : r.statusCode,
                Comparator.comparing(o2 -> o2.statusCode != null ? o2.statusCode.getNumber() : -1)),
            Column.of("Status details",
                r -> r.statusStr,
                Comparator.comparing(o2 -> o2.statusStr))
        );
    }

    private static String parentIdToStr(String id) {
        if (Strings.isNullOrEmpty(id)) {
            return "none";
        }

        return id;
    }

    private static String parentStrToId(String str) {
        if ("none".equals(str)) {
            return "";
        }

        return str;
    }

    public static class Record implements TableRecord {
        String projectId;
        String parentId;
        String alertId;
        int attempt;
        long evaluationLag;
        @Nullable
        EvaluationStatus.Code statusCode;
        String statusStr;
        long ioTime;
        long cpuTime;

        public static Record of(Map.Entry<AlertKey, Task> entry) {
            var r = new Record();
            r.projectId = entry.getKey().getProjectId();
            r.parentId = parentIdToStr(entry.getKey().getParentId());
            r.alertId = entry.getKey().getAlertId();
            r.attempt = entry.getValue().getAttempt();
            r.evaluationLag =  System.currentTimeMillis() - entry.getValue().getEvaluateAt();
            if (r.evaluationLag < 0) {
                r.evaluationLag = 0;
            }

            EvaluationState state = entry.getValue().getState();
            r.statusCode = state != null ? state.getStatus().getCode() : null;
            r.statusStr = state != null ? state.getStatus().getDescription() : "";
            r.ioTime = rate(entry.getValue().getRule().getIoTime());
            r.cpuTime = rate(entry.getValue().getRule().getCpuTime());
            return r;
        }

        private static long rate(Meter meter) {
            return (long) Math.floor(meter.getRate(TimeUnit.MINUTES));
        }
    }
}
