package ru.yandex.solomon.alert.executor.local;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.Metric;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.consume.AlertRuleResultConsumer;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.rule.EvaluationState;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class AlertEvaluationStatusMetrics implements AlertRuleResultConsumer, AlertListener {
    private final MetricRegistry registry;
    private final ConcurrentMap<String, AlertEvalStateMetric> metricByAlertId = new ConcurrentHashMap<>();

    public AlertEvaluationStatusMetrics(MetricRegistry registry) {
        this.registry = registry;
    }

    @Override
    public void consume(Alert alert, EvaluationState state) {
        AlertEvalStateMetric metric = metricByAlertId.get(alert.getId());
        if (metric == null) {
            return;
        }

        metric.update(state);
    }

    @Override
    public void alertChanged(Alert alert) {
    }

    @Override
    public void assignAlert(Alert alert) {
        MetricId metricId = makeMetricId(alert);
        AlertEvalStateMetric metric = (AlertEvalStateMetric)
                registry.putMetricIfAbsent(metricId, new AlertEvalStateMetric(metricId));
        metricByAlertId.put(alert.getId(), metric);
    }

    @Override
    public void unassignAlert(String alertId) {
        AlertEvalStateMetric metric = metricByAlertId.remove(alertId);
        if (metric == null) {
            return;
        }

        registry.removeMetric(metric.getId());
    }

    private MetricId makeMetricId(Alert alert) {
        LabelsBuilder builder = new LabelsBuilder(3)
                .add("alertId", alert.getId())
                .add("projectId", alert.getProjectId());

        if (alert.getAlertType() == AlertType.SUB_ALERT) {
            builder.add("parentId", ((SubAlert) alert).getParent().getId());
        }

        return new MetricId("alert.evaluation.status", builder.build());
    }

    private static class AlertEvalStateMetric implements Metric {
        private final MetricId id;
        private volatile EvaluationState state;

        private AlertEvalStateMetric(MetricId id) {
            this.id = id;
        }

        public MetricId getId() {
            return id;
        }

        public void update(EvaluationState next) {
            this.state = next;
        }

        @Override
        public MetricType type() {
            return MetricType.IGAUGE;
        }

        @Override
        public void accept(long tsMillis, MetricConsumer consumer) {
            EvaluationState actual = state;
            if (actual == null) {
                return;
            }

            consumer.onLong(actual.getLatestEval().toEpochMilli(), actual.getStatus().getCode().getNumber());
        }
    }
}
