package ru.yandex.solomon.alert.executor.local;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ThreadSafe
@ParametersAreNonnullByDefault
public class LocalAlertExecutorOptions {
    private static final LocalAlertExecutorOptions EMPTY = LocalAlertExecutorOptions.newBuilder().build();

    private final long alertRuleTimeoutMillis;
    private final long evalIntervalMillis;
    private final int maxConcurrentAlertRules;
    private final int maxConcurrentWarmupAlertRules;
    private final int maxAlertRules;
    private final long maxEvaluationLagMillis;
    private final long maxEvaluationJitterMillis;

    public LocalAlertExecutorOptions(Builder builder) {
        this.alertRuleTimeoutMillis = builder.alertRuleTimeoutMillis;
        this.evalIntervalMillis = builder.evalIntervalMillis;
        this.maxConcurrentAlertRules = builder.maxConcurrentAlertRules;
        this.maxConcurrentWarmupAlertRules = builder.maxConcurrentWarmupAlertRules;
        this.maxAlertRules = builder.maxAlertRules;
        this.maxEvaluationLagMillis = builder.maxEvaluationLagMillis;
        this.maxEvaluationJitterMillis = builder.maxEvaluationJitterMillis;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static LocalAlertExecutorOptions empty() {
        return EMPTY;
    }

    public long getAlertRuleTimeoutMillis() {
        return alertRuleTimeoutMillis;
    }

    public long getEvalIntervalMillis() {
        return evalIntervalMillis;
    }

    public int getMaxConcurrentAlertRules() {
        return maxConcurrentAlertRules;
    }

    public int getMaxConcurrentWarmupAlertRules() {
        return maxConcurrentWarmupAlertRules;
    }

    public int getMaxAlertRules() {
        return maxAlertRules;
    }

    public long getMaxEvaluationLagMillis() {
        return maxEvaluationLagMillis;
    }

    public long getMaxEvaluationJitterMillis() {
        return maxEvaluationJitterMillis;
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    @ParametersAreNonnullByDefault
    public static class Builder {
        private long alertRuleTimeoutMillis = TimeUnit.SECONDS.toMillis(30);
        private long evalIntervalMillis = TimeUnit.MINUTES.toMillis(1);
        private int maxConcurrentAlertRules = 5;
        private int maxConcurrentWarmupAlertRules = 5;
        private int maxAlertRules = 0;
        private long maxEvaluationLagMillis = TimeUnit.HOURS.toMillis(5L);
        private long maxEvaluationJitterMillis = TimeUnit.SECONDS.toMillis(5L);

        private Builder() {
        }

        private Builder(LocalAlertExecutorOptions options) {
            this.alertRuleTimeoutMillis = options.alertRuleTimeoutMillis;
            this.evalIntervalMillis = options.evalIntervalMillis;
            this.maxConcurrentAlertRules = options.maxConcurrentAlertRules;
            this.maxConcurrentWarmupAlertRules = options.maxConcurrentWarmupAlertRules;
            this.maxAlertRules = options.maxAlertRules;
            this.maxEvaluationLagMillis = options.maxEvaluationLagMillis;
            this.maxEvaluationJitterMillis = options.maxEvaluationJitterMillis;
        }


        public Builder setAlertRuleTimeout(long alertRuleTimeout, TimeUnit unit) {
            this.alertRuleTimeoutMillis = unit.toMillis(alertRuleTimeout);
            return this;
        }

        public Builder setEvalInterval(long evalIntervalMillis, TimeUnit unit) {
            this.evalIntervalMillis = unit.toMillis(evalIntervalMillis);
            return this;
        }

        public Builder setMaxConcurrentAlertRules(int maxConcurrentAlertRules) {
            this.maxConcurrentAlertRules = maxConcurrentAlertRules;
            return this;
        }

        public Builder setMaxAlertRules(int maxAlertRules) {
            this.maxAlertRules = maxAlertRules;
            return this;
        }

        public Builder setMaxConcurrentWarmupAlertRules(int maxConcurrentWarmupAlertRules) {
            this.maxConcurrentWarmupAlertRules = maxConcurrentWarmupAlertRules;
            return this;
        }

        public Builder setMaxEvaluationLag(long value, TimeUnit unit) {
            this.maxEvaluationLagMillis = unit.toMillis(value);
            return this;
        }

        public Builder setMaxEvaluationJitter(long value, TimeUnit unit) {
            this.maxEvaluationJitterMillis = unit.toMillis(value);
            return this;
        }

        public LocalAlertExecutorOptions build() {
            return new LocalAlertExecutorOptions(this);
        }
    }
}
