package ru.yandex.solomon.alert.graph;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.rule.AlertRuleDeadlines;
import ru.yandex.solomon.alert.rule.AlertRuleDefaultDeadlines;
import ru.yandex.solomon.alert.rule.AlertTimeSeries;
import ru.yandex.solomon.alert.rule.ProgramCompiler;
import ru.yandex.solomon.alert.rule.SingleTimeSeries;
import ru.yandex.solomon.alert.rule.threshold.PreparedTransformer;
import ru.yandex.solomon.alert.rule.threshold.Transformer;
import ru.yandex.solomon.alert.util.Converters;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.exceptions.TooManyMetricsLoadedBySelectors;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.solomon.util.time.Interval;
import ru.yandex.stockpile.api.EStockpileStatusCode;

import static ru.yandex.solomon.alert.rule.AlertRuleSelectors.enrichProjectSelector;
import static ru.yandex.solomon.alert.rule.AlertRuleSelectors.toOldFormat;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class GraphLoaderImpl implements GraphLoader {
    private final MetricsClient metricsClient;
    private final ProgramCompiler compiler;

    public GraphLoaderImpl(MetricsClient metricsClient) {
        this.metricsClient = metricsClient;
        this.compiler = new ProgramCompiler();
    }

    @Override
    public CompletableFuture<List<AlertTimeSeries>> load(ThresholdAlert alert, Instant from, Instant to, Instant deadline) {
        AlertRuleDeadlines deadlines = AlertRuleDefaultDeadlines.of(deadline);

        long fromMillis = adjustTimestamp(from.toEpochMilli(), InstantUtils.NOT_BEFORE);
        long toMillis = adjustTimestamp(to.toEpochMilli(), InstantUtils.NOT_AFTER);

        Transformer transformer = new Transformer(alert.getTransformations(), compiler.compile(alert));
        PreparedTransformer prepared = transformer.prepare(Interval.millis(fromMillis, toMillis));
        GraphDataLoadRequest loadRequest = prepared.getLoadRequest();

        Selectors selectors = toOldFormat(enrichProjectSelector(alert, alert.getSelectors()));

        return metricsClient.findAndReadMany(FindAndReadManyRequest.newBuilder()
                .setSelectors(selectors)
                .setMetabaseLimit(alert.getMetricsLimit())
                .setFromMillis(fromMillis)
                .setToMillis(toMillis)
                .addOperation(Operation.newBuilder()
                        .setDownsampling(OperationDownsampling.newBuilder()
                                .setGridMillis(loadRequest.getGridMillis())
                                .setAggregation(Converters.aggregateFunctionToProto(loadRequest.getAggregateFunction()))
                                .setFillOption(OperationDownsampling.FillOption.NULL))
                        .build())
                .setProducer(RequestProducer.SYSTEM)
                .setSoftDeadline(deadlines.softResolveDeadline())
                .setSoftReadDeadline(deadlines.softReadDeadline())
                .setDeadline(deadlines.hardDeadline())
                .build())
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        Throwable ex = CompletableFutures.unwrapCompletionException(throwable);
                        if (ex instanceof TooManyMetricsLoadedBySelectors) {
                            throw new RuntimeException("Too many lines for selectors: " + alert.getSelectors());
                        }
                        if (ex instanceof RuntimeException) {
                            throw (RuntimeException) ex;
                        }
                        throw new RuntimeException(ex);
                    }
                    MetabaseStatus metaStatus = response.getMetaStatus();
                    StockpileStatus storageStatus = response.getStorageStatus();
                    if (!metaStatus.isOkOrNoData()) {
                        throw new RuntimeException("Metabase find response is not ok: " + metaStatus);
                    }
                    if (storageStatus.getCode() != EStockpileStatusCode.OK) {
                        throw new StockpileClientException(storageStatus);
                    }

                    return transform(prepared, response.getMetrics());
                });
    }

    private static List<AlertTimeSeries> transform(PreparedTransformer transformer, List<Metric<MetricKey>> metrics) {
        List<SingleTimeSeries> loadedSeries = metrics.stream()
                .map(metric -> new SingleTimeSeries(metric.getKey(), metric.getType(), metric.getTimeseries()))
                .collect(Collectors.toList());
        return transformer.transform(loadedSeries);
    }

    private long adjustTimestamp(long tsMillis, long replacement) {
        if (InstantUtils.isGoodMillis(tsMillis)) {
            return tsMillis;
        }
        return replacement;
    }
}
