package ru.yandex.solomon.alert.grpc;

import java.util.concurrent.CompletableFuture;

import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.alert.cluster.AlertingShardProxy;
import ru.yandex.solomon.alert.evaluation.EvaluationExplainService;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsResponse;
import ru.yandex.solomon.alert.protobuf.TAlertServiceGrpc;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TCreateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertResponse;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertResponse;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertResponse;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionResponse;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;
import static ru.yandex.solomon.alert.api.ErrorClassificatory.classifyError;
import static ru.yandex.solomon.alert.api.validators.AlertValidator.ensureValid;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcAlertService extends TAlertServiceGrpc.TAlertServiceImplBase {
    private static final Logger logger = LoggerFactory.getLogger(GrpcAlertService.class);
    private final AlertingShardProxy proxy;
    private final EvaluationExplainService explainService;

    // TODO: gather metrics (gordiychuk@)
    public GrpcAlertService(AlertingShardProxy proxy, EvaluationExplainService explainService) {
        this.proxy = proxy;
        this.explainService = explainService;
    }

    @Override
    public void createAlert(TCreateAlertRequest request, StreamObserver<TCreateAlertResponse> responseObserver) {
        asyncComplete(proxy.createAlert(request), responseObserver);
    }

    @Override
    public void updateAlert(TUpdateAlertRequest request, StreamObserver<TUpdateAlertResponse> responseObserver) {
        asyncComplete(proxy.updateAlert(request), responseObserver);
    }

    @Override
    public void deleteAlert(TDeleteAlertRequest request, StreamObserver<TDeleteAlertResponse> responseObserver) {
        asyncComplete(proxy.deleteAlert(request), responseObserver);
    }

    @Override
    public void readAlert(TReadAlertRequest request, StreamObserver<TReadAlertResponse> responseObserver) {
        asyncComplete(proxy.readAlert(request), responseObserver);
    }

    @Override
    public void readAlertInterpolated(TReadAlertInterpolatedRequest request, StreamObserver<TReadAlertInterpolatedResponse> responseObserver) {
        asyncComplete(proxy.readAlert(request), responseObserver);
    }

    @Override
    public void readSubAlert(TReadSubAlertRequest request, StreamObserver<TReadSubAlertResponse> responseObserver) {
        asyncComplete(proxy.readSubAlert(request), responseObserver);
    }

    @Override
    public void listAlert(TListAlertRequest request, StreamObserver<TListAlertResponse> responseObserver) {
        asyncComplete(proxy.listAlerts(request), responseObserver);
    }

    @Override
    public void listSubAlert(TListSubAlertRequest request, StreamObserver<TListSubAlertResponse> responseObserver) {
        asyncComplete(proxy.listSubAlerts(request), responseObserver);
    }

    @Override
    public void readEvaluationState(TReadEvaluationStateRequest request, StreamObserver<TReadEvaluationStateResponse> responseObserver) {
        asyncComplete(proxy.readEvaluationState(request), responseObserver);
    }

    @Override
    public void readEvaluationStats(TReadEvaluationStatsRequest request, StreamObserver<TReadEvaluationStatsResponse> responseObserver) {
        asyncComplete(proxy.readEvaluationStats(request), responseObserver);
    }

    @Override
    public void readNotificationState(TReadNotificationStateRequest request, StreamObserver<TReadNotificationStateResponse> responseObserver) {
        asyncComplete(proxy.readNotificationState(request), responseObserver);
    }

    @Override
    public void readNotificationStats(TReadNotificationStatsRequest request, StreamObserver<TReadNotificationStatsResponse> responseObserver) {
        asyncComplete(proxy.readNotificationStats(request), responseObserver);
    }

    @Override
    public void explainEvaluation(TExplainEvaluationRequest request, StreamObserver<TExplainEvaluationResponse> responseObserver) {
        asyncComplete(explain(request), responseObserver);
    }

    @Override
    public void simulateEvaluation(TSimulateEvaluationRequest request, StreamObserver<TSimulateEvaluationResponse> responseObserver) {
        asyncComplete(simulate(request), responseObserver);
    }

    @Override
    public void readProjectStats(TReadProjectStatsRequest request, StreamObserver<TReadProjectStatsResponse> responseObserver) {
        asyncComplete(proxy.readProjectStats(request), responseObserver);
    }

    @Override
    public void updateAlertTemplateVersion(UpdateAlertTemplateVersionRequest request, StreamObserver<UpdateAlertTemplateVersionResponse> responseObserver) {
        asyncComplete(proxy.updateAlertTemplateVersion(request), responseObserver);
    }

    @Override
    public void createAlertsFromTemplate(CreateAlertsFromTemplateRequest request, StreamObserver<CreateAlertsFromTemplateResponse> responseObserver) {
        asyncComplete(proxy.createAlerts(request), responseObserver);
    }

    @Override
    public void listAlertLabels(ListAlertLabelsRequest request, StreamObserver<ListAlertLabelsResponse> responseObserver) {
        asyncComplete(proxy.listAlertLabels(request), responseObserver);
    }

    private CompletableFuture<TExplainEvaluationResponse> explain(TExplainEvaluationRequest request) {
        CompletableFuture<TExplainEvaluationResponse> future;
        try {
            ensureValid(request);
            future = explainService.explainEvaluation(request);
        } catch (Throwable e) {
            future = CompletableFuture.failedFuture(e);
        }

        return future.exceptionally(e -> {
            ERequestStatusCode code = classifyError(e);
            logger.error("{} - status for explainEvaluation {}", code, request, e);

            return TExplainEvaluationResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
        });
    }

    private CompletableFuture<TSimulateEvaluationResponse> simulate(TSimulateEvaluationRequest request) {
        CompletableFuture<TSimulateEvaluationResponse> future;
        try {
            ensureValid(request);
            future = explainService.simulateEvaluation(request);
        } catch (Throwable e) {
            future = CompletableFuture.failedFuture(e);
        }

        return future.exceptionally(e -> {
            ERequestStatusCode code = classifyError(e);
            logger.error("{} - status for simulateEvaluation {}", code, request, e);

            return TSimulateEvaluationResponse.newBuilder()
                .setRequestStatus(code)
                .setStatusMessage(e.getMessage())
                .build();
        });
    }
}
