package ru.yandex.solomon.alert.grpc;

import java.util.concurrent.CompletableFuture;

import io.grpc.stub.StreamObserver;

import ru.yandex.solomon.alert.api.ErrorClassificatory;
import ru.yandex.solomon.alert.protobuf.AlertTemplateServiceGrpc;
import ru.yandex.solomon.alert.protobuf.CreateAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.DeleteAlertTemplatePublicationRequest;
import ru.yandex.solomon.alert.protobuf.DeleteAlertTemplatePublicationResponse;
import ru.yandex.solomon.alert.protobuf.DeployAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.DeployAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateVersionsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateVersionsResponse;
import ru.yandex.solomon.alert.protobuf.PublishAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.PublishAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ReadAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ReadAlertTemplateResponse;
import ru.yandex.solomon.alert.template.AlertTemplateService;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;
import static ru.yandex.solomon.alert.api.validators.AlertValidator.ensureValid;

/**
 * @author Alexey Trushkin
 */
public class GrpcAlertTemplateService extends AlertTemplateServiceGrpc.AlertTemplateServiceImplBase {
    private final AlertTemplateService alertTemplateService;

    public GrpcAlertTemplateService(AlertTemplateService alertTemplateService) {
        this.alertTemplateService = alertTemplateService;
    }

    @Override
    public void readAlertTemplate(ReadAlertTemplateRequest request, StreamObserver<ReadAlertTemplateResponse> responseObserver) {
        asyncComplete(readAlertTemplate(request), responseObserver);
    }

    @Override
    public void listAlertTemplate(ListAlertTemplateRequest request, StreamObserver<ListAlertTemplateResponse> responseObserver) {
        asyncComplete(listAlertTemplate(request), responseObserver);
    }

    @Override
    public void publishAlertTemplate(PublishAlertTemplateRequest request, StreamObserver<PublishAlertTemplateResponse> responseObserver) {
        asyncComplete(publishAlertTemplate(request), responseObserver);
    }

    @Override
    public void deployAlertTemplate(DeployAlertTemplateRequest request, StreamObserver<DeployAlertTemplateResponse> responseObserver) {
        asyncComplete(deployAlertTemplate(request), responseObserver);
    }

    @Override
    public void createAlertTemplate(CreateAlertTemplateRequest request, StreamObserver<CreateAlertTemplateResponse> responseObserver) {
        asyncComplete(createAlertTemplate(request), responseObserver);
    }

    @Override
    public void deleteAlertTemplatePublication(DeleteAlertTemplatePublicationRequest request, StreamObserver<DeleteAlertTemplatePublicationResponse> responseObserver) {
        asyncComplete(deleteAlertTemplatePublication(request), responseObserver);
    }

    @Override
    public void listAlertTemplateVersions(ListAlertTemplateVersionsRequest request, StreamObserver<ListAlertTemplateVersionsResponse> responseObserver) {
        asyncComplete(listAlertTemplateVersions(request), responseObserver);
    }

    private CompletableFuture<ListAlertTemplateVersionsResponse> listAlertTemplateVersions(ListAlertTemplateVersionsRequest request) {
        try {
            ensureValid(request);
            return alertTemplateService.listTemplateVersions(request)
                    .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(ErrorClassificatory.convertExceptionToGrpc(e));
        }
    }

    private CompletableFuture<DeleteAlertTemplatePublicationResponse> deleteAlertTemplatePublication(DeleteAlertTemplatePublicationRequest request) {
        try {
            ensureValid(request);
            return alertTemplateService.unpublish(request)
                    .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(ErrorClassificatory.convertExceptionToGrpc(e));
        }
    }

    private CompletableFuture<PublishAlertTemplateResponse> publishAlertTemplate(PublishAlertTemplateRequest request) {
        try {
            ensureValid(request);
            return alertTemplateService.publish(request)
                    .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(ErrorClassificatory.convertExceptionToGrpc(e));
        }
    }

    private CompletableFuture<DeployAlertTemplateResponse> deployAlertTemplate(DeployAlertTemplateRequest request) {
        try {
            ensureValid(request);
            return alertTemplateService.deploy(request)
                    .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(ErrorClassificatory.convertExceptionToGrpc(e));
        }
    }

    private CompletableFuture<ReadAlertTemplateResponse> readAlertTemplate(ReadAlertTemplateRequest request) {
        try {
            ensureValid(request);
            return alertTemplateService.read(request)
                    .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(ErrorClassificatory.convertExceptionToGrpc(e));
        }
    }

    private CompletableFuture<ListAlertTemplateResponse> listAlertTemplate(ListAlertTemplateRequest request) {
        try {
            ensureValid(request);
            return alertTemplateService.list(request)
                    .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(ErrorClassificatory.convertExceptionToGrpc(e));
        }
    }

    private CompletableFuture<CreateAlertTemplateResponse> createAlertTemplate(CreateAlertTemplateRequest request) {
        try {
            ensureValid(request);
            return alertTemplateService.create(request)
                    .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(ErrorClassificatory.convertExceptionToGrpc(e));
        }
    }
}
