package ru.yandex.solomon.alert.grpc;

import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Context;
import io.grpc.Deadline;
import io.grpc.stub.StreamObserver;

import ru.yandex.solomon.alert.cluster.AlertingShardProxy;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.CreateMuteResponse;
import ru.yandex.solomon.alert.protobuf.DeleteMuteRequest;
import ru.yandex.solomon.alert.protobuf.DeleteMuteResponse;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesResponse;
import ru.yandex.solomon.alert.protobuf.MuteServiceGrpc;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsResponse;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteResponse;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class GrpcMuteService extends MuteServiceGrpc.MuteServiceImplBase {
    private final AlertingShardProxy proxy;

    public GrpcMuteService(AlertingShardProxy proxy) {
        this.proxy = proxy;
    }

    private long getDeadline() {
        Deadline deadline = Context.current().getDeadline();
        if (deadline == null) {
            return 0;
        }
        return System.currentTimeMillis() + deadline.timeRemaining(TimeUnit.MILLISECONDS);
    }

    @Override
    public void createMute(CreateMuteRequest request, StreamObserver<CreateMuteResponse> responseObserver) {
        asyncComplete(proxy.createMute(request, getDeadline()), responseObserver);
    }

    @Override
    public void readMute(ReadMuteRequest request, StreamObserver<ReadMuteResponse> responseObserver) {
        asyncComplete(proxy.readMute(request, getDeadline()), responseObserver);
    }

    @Override
    public void updateMute(UpdateMuteRequest request, StreamObserver<UpdateMuteResponse> responseObserver) {
        asyncComplete(proxy.updateMute(request, getDeadline()), responseObserver);
    }

    @Override
    public void deleteMute(DeleteMuteRequest request, StreamObserver<DeleteMuteResponse> responseObserver) {
        asyncComplete(proxy.deleteMute(request, getDeadline()), responseObserver);
    }

    @Override
    public void listMutes(ListMutesRequest request, StreamObserver<ListMutesResponse> responseObserver) {
        asyncComplete(proxy.listMutes(request, getDeadline()), responseObserver);
    }

    @Override
    public void readMuteStats(ReadMuteStatsRequest request, StreamObserver<ReadMuteStatsResponse> responseObserver) {
        asyncComplete(proxy.readMuteStats(request, getDeadline()), responseObserver);
    }
}
